<?php
if (!defined('ABSPATH')) exit;

// Hook to apply server-side translations before output is sent to the browser
add_action ( 'template_redirect', function () {
	if (is_admin ()) {
		return;
	}

	$raw_request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
	$uri = esc_url_raw( $raw_request_uri );
		
	$settings = get_option ( 'gptranslate_options', [ ] );

	if (empty ( $settings ['serverside_translations'] ) || $settings ['serverside_translations'] != 1)
		return;

	// Skip static assets and admin/API routes
	if (preg_match ( '#\.(ico|png|jpe?g|gif|svg|css|js|woff2?|ttf|eot|mp4|webm)$#i', $uri ) || strpos ( $uri, '/wp-' ) === 0 || strpos ( $uri, '/wp-json/' ) === 0) {
		return;
	}

	function normalizeText($text) {
		$text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
		$text = str_replace(["\xC2\xA0", "\n", "\r", "\t"], ' ', $text); // NBSP e spazi strani
		$text = preg_replace('/\s+/u', ' ', $text); // spazi multipli → singolo spazio
		$text = trim($text);
		return $text;
	}
	
	ob_start ( function ($html) use ($uri, $settings) {
		global $wpdb;

		$originalLang = $settings ['language'] ?? '';

		if (! defined ( 'GPTRANSLATE_CURRENT_LANG' )) {
			return $html;
		}

		$translatedLang = GPTRANSLATE_CURRENT_LANG;
		
		if (empty ( $originalLang ) || $originalLang === $translatedLang)
			return $html;

		$pageLink = rtrim ( get_site_url (), '/' ) . '/' . ltrim ( GPTRANSLATE_CURRENT_LANG, '/' ) . $uri;

		if ( $settings ['serverside_translations_urldecode'] ) {
			$pageLink = urldecode ( $pageLink );
		}
		
		if ($settings ['serverside_translations_ignore_querystring'] == 1) {
			// Remove query string
			$pageLink = strtok ( $pageLink, '?' );
		} elseif (! empty ( $_SERVER ['QUERY_STRING'] )) {
			// Sanitize and rebuild query string
			$raw_qs = isset($_SERVER['QUERY_STRING']) ? wp_unslash($_SERVER['QUERY_STRING']) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			
			parse_str ( $raw_qs, $qs_args );

			$clean_args = [ ];
			foreach ( $qs_args as $key => $value ) {
				$clean_key = sanitize_key ( $key );
				$clean_value = is_array ( $value ) ? array_map ( 'sanitize_text_field', $value ) : sanitize_text_field ( $value );
				$clean_args [$clean_key] = $clean_value;
			}

			$clean_qs = http_build_query ( $clean_args, '', '&', PHP_QUERY_RFC3986 );

			if ($clean_qs !== '') {
				$pageLink = strtok ( $pageLink, '?' ) . '?' . $clean_qs;
			}
		}

		if ( $settings ['serverside_translations_urlencode_space'] ) {
			$pageLink = str_ireplace ( ' ', '%20', $pageLink );
		}

		$table = $wpdb->prefix . 'gptranslate';
		
		if( $settings ['rewrite_language_url'] && $settings ['rewrite_language_alias'] ) {
			$row = $wpdb->get_row ( $wpdb->prepare ( "SELECT pagelink, translations, alt_translations FROM {$wpdb->prefix}gptranslate" . // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
													 "\n WHERE ( pagelink = %s OR pagelink = %s OR translated_alias = %s OR translated_alias = %s) AND languageoriginal = %s AND languagetranslated = %s AND published = 1", 
													 rtrim($pageLink, '/'), rtrim($pageLink, '/') . '/', rtrim($pageLink, '/'), rtrim($pageLink, '/') . '/', $originalLang, $translatedLang ), ARRAY_A );
		} else {
			$row = $wpdb->get_row ( $wpdb->prepare ( "SELECT pagelink, translations, alt_translations FROM {$wpdb->prefix}gptranslate" . // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	        		 								 "\n WHERE ( pagelink = %s OR pagelink = %s ) AND languageoriginal = %s AND languagetranslated = %s AND published = 1", 
	        		 								 rtrim($pageLink, '/'), rtrim($pageLink, '/') . '/', $originalLang, $translatedLang ), ARRAY_A );
		}

		if (! $row) {
			return $html;
		}

		// Load translated aliases to replace on page links
		$translatedAliasesMap = []; // Map absolute URLs
		$translatedAliasesRelativeMap = []; // Map relative URL (pathname)
		
		if (!empty($settings['rewrite_page_links']) && !empty($settings['rewrite_language_url']) && !empty($settings['rewrite_language_alias'])) {
			try {
				$aliasesResult = $wpdb->get_results( $wpdb->prepare(
						"SELECT pagelink, translated_alias" .
						"\n FROM {$table}" .
						"\n WHERE languagetranslated = %s" .
						"\n AND published = 1",
						$translatedLang
				), ARRAY_A );
				
				if ($aliasesResult) {
					$parsedRoot = trailingslashit(get_site_url());
					
					foreach ($aliasesResult as $rowAliasResult) {
						// NORMALIZE, WORDPRESS STYLE (KEEP TRAILING SLASH)
						$pagelink = gpt_trailingslashit_url($rowAliasResult['pagelink']);
						$translatedAlias = $rowAliasResult['translated_alias'] ?? '';
						
						if (empty($translatedAlias)) continue;
						
						$translatedAlias = gpt_trailingslashit_url($translatedAlias);
						
						// ENCODE ABSOLUTE URL
						$encodedPagelink = $pagelink;
						$parsedUrl = parse_url($pagelink);
						if (isset($parsedUrl['path'])) {
							$pathParts = explode('/', $parsedUrl['path']);
							$encodedParts = array_map('rawurlencode', $pathParts);
							$encodedPath = implode('/', $encodedParts);
							
							$encodedPagelink = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
							$encodedPagelink .= $encodedPath;
							if (isset($parsedUrl['query'])) {
								$encodedPagelink .= '?' . $parsedUrl['query'];
							}
							if (isset($parsedUrl['fragment'])) {
								$encodedPagelink .= '#' . $parsedUrl['fragment'];
							}
						}
						
						// ENCODE TRANSLATED ALIAS
						$parsedAlias = parse_url($translatedAlias);
						$encodedAlias = $translatedAlias;
						
						if (isset($parsedAlias['path'])) {
							$pathParts = explode('/', $parsedAlias['path']);
							$encodedParts = array_map('rawurlencode', $pathParts);
							$encodedPath = implode('/', $encodedParts);
							
							$encodedAlias = $parsedAlias['scheme'] . '://' . $parsedAlias['host'];
							$encodedAlias .= $encodedPath;
							if (isset($parsedAlias['query'])) {
								$encodedAlias .= '?' . $parsedAlias['query'];
							}
							if (isset($parsedAlias['fragment'])) {
								$encodedAlias .= '#' . $parsedAlias['fragment'];
							}
						}
						
						// First pass, check in absolute URLs
						$translatedAliasesMap[$encodedPagelink] = $encodedAlias;
						
						// Second pass, check in relative URLs (pathname)
						$relativePath = str_replace($parsedRoot, '/', $pagelink);
						$relativePath = parse_url($relativePath, PHP_URL_PATH);
						if ($relativePath) {
							// Encode the relative path
							$pathParts = explode('/', $relativePath);
							$encodedParts = array_map('rawurlencode', $pathParts);
							$encodedRelativePath = implode('/', $encodedParts);
							
							$translatedAliasesRelativeMap[$encodedRelativePath] = $encodedAlias;
						}
					}
				}
			} catch (Exception $e) {
				// Silently fail
			}
		}

		// Process replacement method
		if ($settings ['serverside_translations_method'] == 'regex') {
			$translations = json_decode ( $row ['translations'], true ) ?? [ ];
			$altTranslations = json_decode ( $row ['alt_translations'], true ) ?? [ ];
			uksort ( $translations, fn ($a, $b) => strlen ( $b ) - strlen ( $a ) );
			uksort ( $altTranslations, fn ($a, $b) => strlen ( $b ) - strlen ( $a ) );

			$caseInsensitive = ! empty ( $settings ['serverside_translations_caseinsensitive'] );
			$matchQuotes = ! empty ( $settings ['serverside_translations_matchquotes'] );
			$excludedPatterns = [ ];

			$excludedCss = preg_replace ( '/,+/', ',', str_ireplace ( [ 
					"\r",
					"\n",
					'"'
			], [ 
					',',
					',',
					''
			], $settings ['css_selector_serverside_leafnodes_excluded'] ?? '') );
			$excludedCss = array_filter ( array_map ( 'trim', explode ( ',', $excludedCss ) ) );

			foreach ( $excludedCss as $selector ) {
				if (preg_match ( '/^([a-z0-9]+)\.(.+)$/i', $selector, $m )) {
					$excludedPatterns [] = '/<' . preg_quote ( $m [1], '/' ) . '(?=[^>]*\sclass\s*=\s*["\'][^"\']*\b' . preg_quote ( $m [2], '/' ) . '\b)[^>]*>/i';
				} elseif (preg_match ( '/^\.(.+)$/', $selector, $m )) {
					$excludedPatterns [] = '/<([a-z0-9]+)(?=[^>]*\sclass\s*=\s*["\'][^"\']*\b' . preg_quote ( $m [1], '/' ) . '\b)[^>]*>/i';
				}
			}

			$segments = preg_split ( '/(<[^>]+>)/i', $html, - 1, PREG_SPLIT_DELIM_CAPTURE );

			$skipStack = [ ];
			foreach ( $segments as $index => $segment ) {
				if (preg_match ( '/^<\s*(script|style)(\s|>)/i', $segment, $matches )) {
					$skipStack [] = strtolower ( $matches [1] );
				} elseif (preg_match ( '/<\/\s*(script|style)[^>]*>/i', $segment, $matches )) { // <--- FIX QUI
					$tag = strtolower ( $matches [1] );
					if (! empty ( $skipStack ) && end ( $skipStack ) === $tag) {
						array_pop ( $skipStack );
					}
				} elseif (preg_match ( '/^<\s*([a-zA-Z0-9]+)/', $segment, $tagMatch )) {
					$tagName = strtolower ( $tagMatch [1] );
					foreach ( $excludedPatterns as $pattern ) {
						if (preg_match ( $pattern, $segment )) {
							$skipStack [] = $tagName;
							break;
						}
					}
				} elseif (preg_match ( '/^<\/\s*([a-zA-Z0-9]+)/', $segment, $tagMatch )) {
					$tagName = strtolower ( $tagMatch [1] );
					if (! empty ( $skipStack ) && end ( $skipStack ) === $tagName) {
						array_pop ( $skipStack );
					}
				} elseif (! preg_match ( '/^<[^>]+>$/', $segment )) {
					if (empty ( $skipStack )) {
						if ($matchQuotes) {
							$segment = str_ireplace ( '"', "'", $segment );
						}
						foreach ( $translations as $originalText => $translatedText ) {
							$originalText = normalizeText($originalText);
							$segment = normalizeText($segment);
							
							$prevSegment = $segment;
							if ($caseInsensitive) {
								$segment = str_ireplace ( trim ( $originalText ), $translatedText, $segment );
							} else {
								$segment = str_replace ( trim ( $originalText ), $translatedText, $segment );
							}
							if ($segment !== $prevSegment) {
								break;
							}
						}
					}
				}

				$segments [$index] = $segment;
			}

			$html = implode ( '', $segments );

			// Replace alt and title in images
			if (! empty ( $settings ['translate_altimages'] )) {
				$html = preg_replace_callback ( '/<img[^>]*\b(alt|title)\s*=\s*([\"\'])(.*?)\2[^>]*>/i', function ($matches) use ($altTranslations, $caseInsensitive, $matchQuotes) {
					$attr = $matches [1];
					$quote = $matches [2];
					$value = $matches [3];
					if ($matchQuotes)
						$value = str_ireplace ( '"', "'", $value );
					foreach ( $altTranslations as $original => $translated ) {
						$original = normalizeText($original);
						$value = normalizeText($value);
						
						$value = $caseInsensitive ? str_ireplace ( trim ( $original ), $translated, $value ) : str_replace ( trim ( $original ), $translated, $value );
					}
					return preg_replace ( '/\b' . $attr . '\s*=\s*["\'].*?["\']/', "$attr=$quote$value$quote", $matches [0] );
				}, $html );
				
				// Check if also images 'src' are enabled to be translated
				if (! empty ( $settings ['translate_srcimages'] )) {
					$html = preg_replace_callback('/<img\b[^>]*\bsrc\s*=\s*(["\'])(.*?)\1[^>]*>/i', function ($matches) use ($altTranslations, $caseInsensitive, $matchQuotes) {
						$originalTag = $matches[0];
						$quote = $matches[1];
						$srcValue = ltrim($matches[2], '/');
						
						if ($matchQuotes) {
							$srcValue = str_ireplace('"', "'", $srcValue);
						}
						
						$translatedSrcValue = null;
						foreach ($altTranslations as $originalText => $translatedText) {
							$originalText = ltrim($originalText, '/');
							if ($caseInsensitive) {
								if (strcasecmp(trim($originalText), $srcValue) === 0) {
									$translatedSrcValue = $translatedText;
									break;
								}
							} else {
								if (trim($originalText) === $srcValue) {
									$translatedSrcValue = $translatedText;
									break;
								}
							}
						}
						
						// If a translation was found, replace src and srcset
						if ($translatedSrcValue !== null) {
							// Replace the src attribute
							$modifiedTag = preg_replace('/\bsrc\s*=\s*["\'].*?["\']/i', 'src=' . $quote . $translatedSrcValue . $quote, $originalTag);
							
							// Check if srcset exists and replace it with the translated src value
							if (preg_match('/\bsrcset\s*=\s*(["\'])(.*?)\1/i', $modifiedTag, $srcsetMatch)) {
								$srcsetQuote = $srcsetMatch[1];
								$modifiedTag = preg_replace('/\bsrcset\s*=\s*["\'].*?["\']/i', 'srcset=' . $srcsetQuote . $translatedSrcValue . $srcsetQuote, $modifiedTag);
							}
							
							return $modifiedTag;
						}
						
						return $originalTag;
					}, $html);
				}
			}
			
			// Translate <meta name="description" content="...">
			$html = preg_replace_callback(
				'~<meta\s+(?:name|property)=["\'](?:description|og:description|twitter:description)["\']\s+content=["\'](.*?)["\'][^>]*>~i',
				function ($matches) use ($altTranslations, $caseInsensitive, $matchQuotes) {
					$originalTag = $matches[0];
					$contentValue = $matches[1];
					
					if ($matchQuotes) {
						$contentValue = str_ireplace('"', "'", $contentValue);
					}
					
					foreach ($altTranslations as $originalText => $translatedText) {
						$prev = $contentValue;
						if ($caseInsensitive) {
							$contentValue = str_ireplace(trim($originalText), $translatedText, $contentValue);
						} else {
							$contentValue = str_replace(trim($originalText), $translatedText, $contentValue);
						}
						if ($contentValue !== $prev) break;
					}
					
					return preg_replace(
						'/content=["\'].*?["\']/i',
						'content="' . $contentValue . '"',
						$originalTag
					);
				},
				$html
			);

			// Add skip marker
			$html = preg_replace ( '/<body/i', '<body data-gptranslateskip="1" data-gptranslateoriginalalias="' . $row['pagelink'] . '"', $html, 1 );
			
			// Replace page links in <a href="..."> tags based on translated aliases
			if (!empty($translatedAliasesMap) || !empty($translatedAliasesRelativeMap)) {
				$html = preg_replace_callback(
						'/<a\s+([^>]*\s)?href\s*=\s*(["\'])(.*?)\2([^>]*)>/i',
						function($matches) use ($translatedAliasesMap, $translatedAliasesRelativeMap, $settings) {
							$fullTag = $matches[0];
							$quote = $matches[2];
							$href = $matches[3];
							
							// Decode href attribute to match database entries
							$decodedHref = html_entity_decode($href, ENT_QUOTES, 'UTF-8');

							// Extract hash/fragment if present
							$hashFragment = '';
							if (strpos($decodedHref, '#') !== false) {
								$parts = explode('#', $decodedHref, 2);
								$decodedHref = $parts[0]; // URL without hash
								$hashFragment = '#' . $parts[1]; // Save the hash
							}

							$queryString = '';
							if ($settings ['ignore_querystring'] == 1 && strpos($decodedHref, '?') !== false) {
								$parts = explode('?', $decodedHref, 2);
								$decodedHref = $parts[0];
								$queryString = '?' . $parts[1];
							}
							$originalPathBeforeNormalization = $decodedHref;
							
							$decodedHref = gpt_trailingslashit_url($decodedHref);
							
							$translatedAlias = null;
							
							// First pass, check in absolute URLs
							if (isset($translatedAliasesMap[$decodedHref])) {
								$translatedAlias = $translatedAliasesMap[$decodedHref];
							}
							// Second pass, check in relative URLs (pathname)
							elseif (isset($translatedAliasesRelativeMap[$decodedHref])) {
								$translatedAlias = $translatedAliasesRelativeMap[$decodedHref];
							}
							
							// If found a translated alias, replace href and add data-originalhref if not present
							if ($translatedAlias) {
								$finalHref = $translatedAlias . $queryString . $hashFragment;

								$originalHrefFull = $originalPathBeforeNormalization . $queryString . $hashFragment;
								
								// Check if data-originalhref already exists in the tag
								if (strpos($fullTag, 'data-originalhref') === false) {
									$dataAttr = ' data-originalhref=' . $quote . htmlspecialchars($originalHrefFull, ENT_QUOTES, 'UTF-8') . $quote;
									$newTag = str_ireplace('<a ', '<a' . $dataAttr . ' ', $fullTag);
									$newTag = str_ireplace('href=' . $quote . $href . $quote, 'href=' . $quote . htmlspecialchars($finalHref, ENT_QUOTES, 'UTF-8') . $quote, $newTag);
									return $newTag;
								} else {
									return str_ireplace('href=' . $quote . $href . $quote, 'href=' . $quote . htmlspecialchars($finalHref, ENT_QUOTES, 'UTF-8') . $quote, $fullTag);
								}
							}
							
							return $fullTag;
						},
					$html
				);
			}
		} elseif ($settings ['serverside_translations_method'] == 'domdocument') {
			// Solution 2: classic DOMDocument approach, effective but could cause closing tags and encoding issues
			try {
				$translationsArray = json_decode ( $row['translations'], true ) ?? [ ];
				$altTranslationsArray = json_decode ( $row['alt_translations'] ?? '', true ) ?: [ ];
				
				// Sort the translation keys in descending order by length.
				uksort ( $translationsArray, function ( $a, $b ) {
					return strlen ( $b ) - strlen ( $a );
				} );
					
				// Sort the alt translation keys in descending order by length.
				uksort ( $altTranslationsArray, function ( $a, $b ) {
					return strlen ( $b ) - strlen ( $a );
				} );
					
				// Initialize DOMDocument without converting the body content via mb_convert_encoding.
				$doc = new DOMDocument ();
				libxml_use_internal_errors ( true );
				// Use an XML encoding hack to inform DOMDocument about UTF-8 while preserving original characters.
				$doc->loadHTML ( '<?xml encoding="UTF-8">' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
				libxml_clear_errors ();
				
				// Use XPath to locate text nodes, skipping those in <script> or <style> tags.
				$xpath = new DOMXPath ( $doc );
				$cssSelectorLeafnodesExcluded = str_ireplace ( '"', '', trim ( trim ( preg_replace ( '/,+/', ',', str_ireplace ( [ "\r", "\n" ], ",", $settings['css_selector_serverside_leafnodes_excluded'] ?? '' ) ) ), ',' ) );
				$excludedNodes = [ ];
				
				if (! empty ( $cssSelectorLeafnodesExcluded )) {
					$selectors = explode ( ',', $cssSelectorLeafnodesExcluded );
					$xpathQueries = [ ];
					
					foreach ( $selectors as $selector ) {
						$selector = trim ( $selector );
						if (empty ( $selector ))
							continue;
							
							// Convert CSS selector to XPath
							$selector = preg_replace ( '/\s+/', ' ', $selector ); // Normalize spaces
							$selectorParts = explode ( ' ', $selector );
							
							$xpathQuery = '';
							foreach ( $selectorParts as $part ) {
								if (preg_match ( '/^([a-zA-Z0-9_-]+)?(\.[a-zA-Z0-9_-]+)?(#[a-zA-Z0-9_-]+)?$/', $part, $matches )) {
									$tag = ! empty ( $matches[1] ) ? $matches[1] : '*'; // If no tag, use '*'
									$class = ! empty ( $matches[2] ) ? substr ( $matches[2], 1 ) : ''; // Remove leading '.'
									$id = ! empty ( $matches[3] ) ? substr ( $matches[3], 1 ) : ''; // Remove leading '#'
									
									$conditions = [ ];
									if ($class) {
										$conditions[] = "contains(concat(' ', normalize-space(@class), ' '), ' $class ')";
									}
									if ($id) {
										$conditions[] = "@id='$id'";
									}
									
									$xpathQuery .= "//{$tag}" . (! empty ( $conditions ) ? "[" . implode ( " and ", $conditions ) . "]" : "");
								}
							}
							
							if (! empty ( $xpathQuery )) {
								$xpathQueries[] = $xpathQuery;
							}
					}
					
					// Execute all XPath queries and collect excluded nodes
					foreach ( $xpathQueries as $query ) {
						foreach ( $xpath->query ( $query ) as $excludedNode ) {
							$excludedNodes[] = $excludedNode;
						}
					}
				}
				
				$textNodes = $xpath->query ( '//text()[not(ancestor::script) and not(ancestor::style) and normalize-space()]' );
				$caseInsensitive = ! empty ( $settings['serverside_translations_caseinsensitive'] );
				$matchQuotes = ! empty ( $settings['serverside_translations_matchquotes'] );
				
				// Process each text node with translation replacements.
				foreach ( $textNodes as $textNode ) {
					if ($cssSelectorLeafnodesExcluded) {
						$skipNode = false;
						foreach ( $excludedNodes as $excludedNode ) {
							if ($excludedNode->isSameNode ( $textNode->parentNode )) {
								$skipNode = true;
								break;
							}
						}
						if ($skipNode)
							continue;
					}
					
					$textContent = $textNode->nodeValue;
					if ($matchQuotes) {
						$textContent = str_ireplace ( '"', "'", $textContent );
					}
					foreach ( $translationsArray as $originalText => $translatedText ) {
						$originalText = normalizeText($originalText);
						$textContent = normalizeText($textContent);
						
						$prevTextContent = $textContent; // Save the current state
						if ($caseInsensitive) {
							$textContent = str_ireplace ( trim ( $originalText ), $translatedText, $textContent );
						} else {
							$textContent = str_replace ( trim ( $originalText ), $translatedText, $textContent );
						}
						// Break out of the loop if a replacement occurred.
						if ($textContent !== $prevTextContent) {
							break;
						}
					}
					$textNode->nodeValue = $textContent;
				}
				
				// Replace page links in <a href="..."> tags based on translated aliases
				if (!empty($translatedAliasesMap) || !empty($translatedAliasesRelativeMap)) {
					$linkNodes = $xpath->query('//a[@href]');
					foreach ($linkNodes as $linkNode) {
						$href = $linkNode->getAttribute('href');
						
						// Decode href attribute (WordPress-style, KEEP trailing slash)
						$decodedHref = html_entity_decode($href, ENT_QUOTES, 'UTF-8');
						
						// Extract hash/fragment if present
						$hashFragment = '';
						if (strpos($decodedHref, '#') !== false) {
							$parts = explode('#', $decodedHref, 2);
							$decodedHref = $parts[0]; // URL without hash
							$hashFragment = '#' . $parts[1]; // Save the hash
						}
						
						$queryString = '';
						if ($settings['ignore_querystring'] == 1 && strpos($decodedHref, '?') !== false) {
							$parts = explode('?', $decodedHref, 2);
							$decodedHref = $parts[0];
							$queryString = '?' . $parts[1];
						}
						$originalPathBeforeNormalization = $decodedHref;
						
						$decodedHref = gpt_trailingslashit_url($decodedHref);
						
						$translatedAlias = null;
						
						// Cerca prima negli URL assoluti
						if (isset($translatedAliasesMap[$decodedHref])) {
							$translatedAlias = $translatedAliasesMap[$decodedHref];
						}
						// Poi cerca negli URL relativi
						elseif (isset($translatedAliasesRelativeMap[$decodedHref])) {
							$translatedAlias = $translatedAliasesRelativeMap[$decodedHref];
						}
						
						// Se trovato un alias tradotto, sostituisci
						if ($translatedAlias) {
							// Aggiungi data-originalhref se non presente
							if (!$linkNode->hasAttribute('data-originalhref')) {
								$linkNode->setAttribute('data-originalhref', $originalPathBeforeNormalization . $queryString . $hashFragment);
							}
							$linkNode->setAttribute('href', $translatedAlias . $queryString . $hashFragment);
						}
					}
				}
				
				// Check if also images 'alt' are enabled to be translated
				if (! empty ( $settings['translate_altimages'] )) {
					$imgNodesAlt = $xpath->query ( '//img[@alt]' );
					foreach ( $imgNodesAlt as $imgNode ) {
						$altText = $imgNode->getAttribute ( 'alt' );
						if ($matchQuotes) {
							$altText = str_ireplace ( '"', "'", $altText );
						}
						foreach ( $altTranslationsArray as $originalText => $translatedText ) {
							$originalText = normalizeText($originalText);
							$altText = normalizeText($altText);
							
							$prevAltText = $altText;
							if ($caseInsensitive) {
								$altText = str_ireplace ( trim ( $originalText ), $translatedText, $altText );
							} else {
								$altText = str_replace ( trim ( $originalText ), $translatedText, $altText );
							}
							if ($altText !== $prevAltText) {
								break;
							}
						}
						$imgNode->setAttribute ( 'alt', $altText );
					}
					
					$imgNodesTitle = $xpath->query ( '//img[@title]' );
					foreach ( $imgNodesTitle as $imgNode ) {
						$titleText = $imgNode->getAttribute ( 'title' );
						if ($matchQuotes) {
							$titleText = str_ireplace ( '"', "'", $titleText );
						}
						foreach ( $altTranslationsArray as $originalText => $translatedText ) {
							$originalText = normalizeText($originalText);
							$titleText = normalizeText($titleText);
							
							$prevTitleText = $titleText;
							if ($caseInsensitive) {
								$titleText = str_ireplace ( trim ( $originalText ), $translatedText, $titleText );
							} else {
								$titleText = str_replace ( trim ( $originalText ), $translatedText, $titleText );
							}
							if ($titleText !== $prevTitleText) {
								break;
							}
						}
						$imgNode->setAttribute ( 'title', $titleText );
					}
					
					// Check if also images 'src' are enabled to be translated
					if (! empty ( $settings ['translate_srcimages'] )) {
						$imgNodesSrc = $xpath->query ( '//img[@src]' );
						foreach ( $imgNodesSrc as $imgNode ) {
							$srcValue = ltrim ( $imgNode->getAttribute ( 'src' ), '/' );
							
							if ($matchQuotes) {
								$srcValue = str_ireplace ( '"', "'", $srcValue );
							}
							
							$translatedSrcValue = null;
							foreach ( $altTranslationsArray as $originalText => $translatedText ) {
								$originalText = normalizeText($originalText);
								$srcValue = normalizeText($srcValue);
								
								$originalText = ltrim ( $originalText, '/' );
								if ($caseInsensitive) {
									if (strcasecmp ( trim ( $originalText ), $srcValue ) === 0) {
										$translatedSrcValue = $translatedText;
										break;
									}
								} else {
									if (trim ( $originalText ) === $srcValue) {
										$translatedSrcValue = $translatedText;
										break;
									}
								}
							}
							
							// If a translation was found, replace src and srcset
							if ($translatedSrcValue !== null) {
								// Replace the src attribute
								$imgNode->setAttribute ( 'src', $translatedSrcValue );
								
								// Check if srcset exists and replace it with the translated src value
								if ($imgNode->hasAttribute ( 'srcset' )) {
									$imgNode->setAttribute ( 'srcset', $translatedSrcValue );
								}
							}
						}
					}
				}
				
				// Translate meta descriptions (standard, Open Graph, Twitter)
				$metaDescriptions = $xpath->query(
					'//meta[
						(@name="description")
						or (@name="twitter:description")
						or (@property="og:description")
					]'
				);
				foreach ($metaDescriptions as $meta) {
					$contentValue = $meta->getAttribute('content');
					
					if ($matchQuotes) {
						$contentValue = str_ireplace('"', "'", $contentValue);
					}
					
					foreach ($altTranslationsArray as $originalText => $translatedText) {
						$prev = $contentValue;
						if ($caseInsensitive) {
							$contentValue = str_ireplace(trim($originalText), $translatedText, $contentValue);
						} else {
							$contentValue = str_replace(trim($originalText), $translatedText, $contentValue);
						}
						if ($contentValue !== $prev) break;
					}
					
					$meta->setAttribute('content', $contentValue);
				}
				
				// Optionally, add an attribute to the <body> tag to signal that translations were applied.
				$bodyTag = $doc->getElementsByTagName ( 'body' )->item ( 0 );
				if ($bodyTag) {
					$bodyTag->setAttribute ( 'data-gptranslateskip', '1' );
					$bodyTag->setAttribute ( 'data-gptranslateoriginalalias', htmlspecialchars($row['pagelink'], ENT_QUOTES, 'UTF-8') );
				}
				
				// Save the modified HTML.
				$html = $doc->saveHTML ();
				
				// Decode HTML entities back into UTF-8 characters.
				$html = html_entity_decode ( $html, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
	
			} catch ( Exception $e ) {
				// Handle exception if needed
			}
		} elseif ($settings['serverside_translations_method'] == 'simplehtmldom') {
			require_once plugin_dir_path(__FILE__) . 'simplehtmldom.php';
			
			// Helper: checks if $child is inside (or is) $parent.
			function nodeIsInsideExcluded($child, $excludedNodes) {
				while ($child !== null) {
					foreach ($excludedNodes as $ex) {
						if ($child === $ex) {
							return true;
						}
					}
					$child = $child->parent;
				}
				return false;
			}
			
			// Recursive function to process all text nodes.
			function processTextNodes($node, $excludedNodes, $translationsArray, $caseInsensitive, $matchQuotes) {
				if ($node->tag === 'text') {
					if (! nodeIsInsideExcluded($node, $excludedNodes) && trim($node->innertext) && $node->innertext != "\t") {
						$text = $node->innertext;
						if ($matchQuotes) {
							$text = str_ireplace('"', "'", $text);
						}
						
						// Apply translations without overriding longer translations
						$processedParts = [];
						foreach ($translationsArray as $originalText => $translatedText) {
							$originalText = normalizeText($originalText);
							$text = normalizeText($text);
							
							if ($caseInsensitive) {
								$text = preg_replace_callback(
										'/(?<!\w)' . preg_quote(trim($originalText), '/') . '(?!\w)/ui',
										function ($matches) use ($translatedText, &$processedParts) {
											if (in_array($matches[0], $processedParts, true)) {
												return $matches[0];
											}
											$processedParts[] = $translatedText;
											return $translatedText;
										},
										$text
										);
							} else {
								$text = preg_replace_callback(
										'/(?<!\w)' . preg_quote(trim($originalText), '/') . '(?!\w)/',
										function ($matches) use ($translatedText, &$processedParts) {
											if (in_array($matches[0], $processedParts, true)) {
												return $matches[0];
											}
											$processedParts[] = $translatedText;
											return $translatedText;
										},
										$text
										);
							}
						}
						$node->innertext = $text;
					}
				} else {
					$tagLower = strtolower($node->tag ?? '');
					if (in_array($tagLower, ['script', 'style'])) {
						return;
					}
					if (isset($node->nodes) && is_array($node->nodes)) {
						foreach ($node->nodes as $child) {
							processTextNodes($child, $excludedNodes, $translationsArray, $caseInsensitive, $matchQuotes);
						}
					}
				}
			}
			
			// Main processing code.
			try {
				$translationsArray = json_decode($row['translations'], true) ?? [];
				$altTranslationsArray = json_decode($row['alt_translations'] ?? '', true) ?: [];
				
				// Sort translations by descending key length
				uksort($translationsArray, function ($a, $b) {
					return strlen($b) - strlen($a);
				});
					
				// Sort alt translations by descending key length
				uksort($altTranslationsArray, function ($a, $b) {
					return strlen($b) - strlen($a);
				});
				
				$htmlObj = gptranslate_simplehtmldom_str_get_html($html);
				
				$cssSelectorLeafnodesExcluded = str_ireplace(
						'"',
						'',
						trim(trim(preg_replace('/,+/', ',', str_ireplace(["\r", "\n"], ",", $settings['css_selector_serverside_leafnodes_excluded'] ?? ''))), ',')
						);
				
				$excludedNodes = [];
				if (! empty($cssSelectorLeafnodesExcluded)) {
					$selectors = explode(',', $cssSelectorLeafnodesExcluded);
					foreach ($selectors as $selector) {
						$selector = trim($selector);
						if (! empty($selector)) {
							$foundNodes = $htmlObj->find($selector);
							foreach ($foundNodes as $node) {
								$excludedNodes[] = $node;
							}
						}
					}
				}
				
				$caseInsensitive = ! empty($settings['serverside_translations_caseinsensitive']);
				$matchQuotes = ! empty($settings['serverside_translations_matchquotes']);
				
				processTextNodes($htmlObj, $excludedNodes, $translationsArray, $caseInsensitive, $matchQuotes);
				
				// Replace page links in <a href="..."> tags based on translated aliases
				if (!empty($translatedAliasesMap) || !empty($translatedAliasesRelativeMap)) {
					foreach ($htmlObj->find('a[href]') as $linkNode) {
						$href = $linkNode->href;
						
						// Decode href attribute (WordPress-style, KEEP trailing slash)
						$decodedHref = html_entity_decode($href, ENT_QUOTES, 'UTF-8');

						// Extract hash/fragment if present
						$hashFragment = '';
						if (strpos($decodedHref, '#') !== false) {
							$parts = explode('#', $decodedHref, 2);
							$decodedHref = $parts[0]; // URL without hash
							$hashFragment = '#' . $parts[1]; // Save the hash
						}

						// Extract query string if ignore_querystring is enabled
						$queryString = '';
						if ($settings['ignore_querystring'] == 1 && strpos($decodedHref, '?') !== false) {
							$parts = explode('?', $decodedHref, 2);
							$decodedHref = $parts[0];
							$queryString = '?' . $parts[1];
						}
						$originalPathBeforeNormalization = $decodedHref;

						$decodedHref = gpt_trailingslashit_url($decodedHref);
						
						$translatedAlias = null;
						
						// Cerca prima negli URL assoluti
						if (isset($translatedAliasesMap[$decodedHref])) {
							$translatedAlias = $translatedAliasesMap[$decodedHref];
						}
						// Poi cerca negli URL relativi
						elseif (isset($translatedAliasesRelativeMap[$decodedHref])) {
							$translatedAlias = $translatedAliasesRelativeMap[$decodedHref];
						}
						
						// Se trovato un alias tradotto, sostituisci
						if ($translatedAlias) {
							// Aggiungi data-originalhref se non presente
							if (!isset($linkNode->{'data-originalhref'})) {
								$linkNode->{'data-originalhref'} = $originalPathBeforeNormalization . $queryString . $hashFragment;
							}
							$linkNode->href = $translatedAlias . $queryString . $hashFragment;
						}
					}
				}
				
				// Check if also images 'alt' are enabled to be translated
				if (! empty($settings['translate_altimages'])) {
					foreach ($htmlObj->find('img[alt]') as $imgNode) {
						$altText = $imgNode->alt;
						if ($matchQuotes) {
							$altText = str_ireplace('"', "'", $altText);
						}
						foreach ($altTranslationsArray as $originalText => $translatedText) {
							$originalText = normalizeText($originalText);
							$altText = normalizeText($altText);
							
							$prevAltText = $altText;
							if ($caseInsensitive) {
								$altText = str_ireplace(trim($originalText), $translatedText, $altText);
							} else {
								$altText = str_replace(trim($originalText), $translatedText, $altText);
							}
							if ($altText !== $prevAltText) {
								break;
							}
						}
						$imgNode->alt = $altText;
					}
					
					foreach ($htmlObj->find('img[title]') as $imgNode) {
						$titleText = $imgNode->title;
						if ($matchQuotes) {
							$titleText = str_ireplace('"', "'", $titleText);
						}
						foreach ($altTranslationsArray as $originalText => $translatedText) {
							$originalText = normalizeText($originalText);
							$titleText = normalizeText($titleText);
							
							$prevTitleText = $titleText;
							if ($caseInsensitive) {
								$titleText = str_ireplace(trim($originalText), $translatedText, $titleText);
							} else {
								$titleText = str_replace(trim($originalText), $translatedText, $titleText);
							}
							if ($titleText !== $prevTitleText) {
								break;
							}
						}
						$imgNode->title = $titleText;
					}
					
					// Check if also images 'src' are enabled to be translated
					if (! empty($settings['translate_srcimages'])) {
						foreach ($htmlObj->find('img[src]') as $imgNode) {
							$srcValue = ltrim($imgNode->src, '/');
							
							if ($matchQuotes) {
								$srcValue = str_ireplace('"', "'", $srcValue);
							}
							
							$translatedSrcValue = null;
							foreach ($altTranslationsArray as $originalText => $translatedText) {
								$originalText = normalizeText($originalText);
								$srcValue = normalizeText($srcValue);
								
								$originalText = ltrim($originalText, '/');
								if ($caseInsensitive) {
									if (strcasecmp(trim($originalText), $srcValue) === 0) {
										$translatedSrcValue = $translatedText;
										break;
									}
								} else {
									if (trim($originalText) === $srcValue) {
										$translatedSrcValue = $translatedText;
										break;
									}
								}
							}
							
							// If a translation was found, replace src and srcset
							if ($translatedSrcValue !== null) {
								// Replace the src attribute
								$imgNode->src = $translatedSrcValue;
								
								// Check if srcset exists and replace it with the translated src value
								if (isset($imgNode->srcset)) {
									$imgNode->srcset = $translatedSrcValue;
								}
							}
						}
					}
				}
				
				// Translate <meta name="description">
				foreach ($htmlObj->find('meta[name=description], meta[name=twitter:description], meta[property=og:description]') as $metaNode) {
					$contentValue = $metaNode->content;
					
					if ($matchQuotes) {
						$contentValue = str_ireplace('"', "'", $contentValue);
					}
					
					foreach ($altTranslationsArray as $originalText => $translatedText) {
						$prev = $contentValue;
						if ($caseInsensitive) {
							$contentValue = str_ireplace(trim($originalText), $translatedText, $contentValue);
						} else {
							$contentValue = str_replace(trim($originalText), $translatedText, $contentValue);
						}
						if ($contentValue !== $prev) break;
					}
					
					$metaNode->content = $contentValue;
				}
				
				if ($bodyElement = $htmlObj->find('body', 0)) {
					$bodyElement->setAttribute('data-gptranslateskip', '1');
					$bodyElement->setAttribute ( 'data-gptranslateoriginalalias', htmlspecialchars($row['pagelink'], ENT_QUOTES, 'UTF-8') );
				}
				
				$modifiedHtml = $htmlObj->save();
				$modifiedHtml = html_entity_decode($modifiedHtml, ENT_QUOTES | ENT_HTML5, 'UTF-8');
				
				$html = $modifiedHtml;
			} catch (Exception $e) {
				// Handle exceptions as needed
			}
		} elseif ($settings['serverside_translations_method'] == 'strireplace') {
			// Solution 3: simplest approach, unconditional str_ireplace that could cause unintentional replacements
			try {
				$translationsArray = json_decode($row['translations'], true) ?? [];
				$altTranslationsArray = json_decode($row['alt_translations'] ?? '', true) ?: [];
				
				// Do body page translations replacements
				uksort($translationsArray, function ($a, $b) {
					return strlen($b) - strlen($a);
				});
					
				$caseInsensitive = !empty($settings['serverside_translations_caseinsensitive']);
				$matchQuotes = !empty($settings['serverside_translations_matchquotes']);
				
				foreach ($translationsArray as $originalText => $translatedText) {
					$originalText = normalizeText($originalText);
					$html = normalizeText($html);
					
					if ($caseInsensitive) {
						$html = str_ireplace(trim($originalText), $translatedText, $html);
						
						// Check also if both single quotes or double quotes should be checked to replace
						if ($matchQuotes && strpos($originalText, "'") !== false) {
							$originalTextAlt = str_ireplace("'", '"', $originalText);
							$html = str_ireplace(trim($originalTextAlt), $translatedText, $html);
						}
					} else {
						$html = str_replace(trim($originalText), $translatedText, $html);
						
						if ($matchQuotes && strpos($originalText, "'") !== false) {
							$originalTextAlt = str_replace("'", '"', $originalText);
							$html = str_replace(trim($originalTextAlt), $translatedText, $html);
						}
					}
				}
				
				// Check if also images 'alt' are enabled to be translated
				if (!empty($settings['translate_altimages'])) {
					foreach ($altTranslationsArray as $originalAlt => $translatedAlt) {
						if ($matchQuotes) {
							$originalAlt = str_ireplace('"', "'", $originalAlt);
						}
						if ($caseInsensitive) {
							$html = str_ireplace('alt="' . trim($originalAlt) . '"', 'alt="' . $translatedAlt . '"', $html);
							$html = str_ireplace("alt='" . trim($originalAlt) . "'", "alt='" . $translatedAlt . "'", $html);
						} else {
							$html = str_replace('alt="' . trim($originalAlt) . '"', 'alt="' . $translatedAlt . '"', $html);
							$html = str_replace("alt='" . trim($originalAlt) . "'", "alt='" . $translatedAlt . "'", $html);
						}
					}
				}
				
				$html = str_ireplace('<body', '<body data-gptranslateskip="1" data-gptranslateoriginalalias="' . $row['pagelink'] . '"', $html);
					
			} catch (Exception $e) {
				// Handle exceptions as needed
			}
		}

		return $html;
	} );
} );
