<?php
/**
 * GPTranslate - Multilang Routing System for WordPress
 * Handles language prefixes in URLs like /it/, /fr/, /en/, etc.
 * and maps them to internal pages, similar to Joomla LanguageFilter plugin.
 */

if (!defined('ABSPATH')) exit;

// Load only in frontend
if (! is_admin ()) {
	$settings = get_option("gptranslate_options");
	
	if ( ! function_exists( 'is_plugin_active' ) ) {
		include_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	$polylangActive = is_plugin_active( 'polylang/polylang.php' );
	
	$enabledLanguages = $settings['languages'] ?? [];
	
	// Define supported languages (lowercase)
	$gpt_supported_languages = array_map ( 'strtolower', [ 
			'AF',
			'SQ',
			'AM',
			'AR',
			'HY',
			'AZ',
			'EU',
			'BE',
			'BN',
			'BS',
			'BG',
			'CA',
			'CEB',
			'NY',
			'ZH',
			'CO',
			'HR',
			'CS',
			'DA',
			'NL',
			'EN',
			'EO',
			'ET',
			'TL',
			'FI',
			'FR',
			'FY',
			'GL',
			'KA',
			'DE',
			'EL',
			'GU',
			'HT',
			'HA',
			'HAW',
			'IW',
			'HI',
			'HMN',
			'HU',
			'IS',
			'IG',
			'ID',
			'GA',
			'IT',
			'JA',
			'JW',
			'KN',
			'KK',
			'KM',
			'KO',
			'KU',
			'KY',
			'LO',
			'LA',
			'LV',
			'LT',
			'LB',
			'MK',
			'MG',
			'MS',
			'ML',
			'MT',
			'MI',
			'MR',
			'MN',
			'MY',
			'NE',
			'NO',
			'PS',
			'FA',
			'PL',
			'PT',
			'PA',
			'RO',
			'RU',
			'SM',
			'GD',
			'SR',
			'ST',
			'SN',
			'SD',
			'SI',
			'SK',
			'SL',
			'SO',
			'ES',
			'SU',
			'SW',
			'SV',
			'TG',
			'TA',
			'TE',
			'TH',
			'TR',
			'UK',
			'UR',
			'UZ',
			'VI',
			'CY',
			'XH',
			'YI',
			'YO',
			'ZU',
			'ZT'
	] );

	/**
	 * GPTranslate - Alias Translation Routing for WordPress
	 * This section augments the language prefix routing by checking translated aliases
	 * and routing them to the original page link.
	 */
	
	add_action('init', function () use ($gpt_supported_languages, $polylangActive, $settings) {
		if (is_admin()) return;
		
		global $wpdb;
		
		$subfolder = $settings['subfolder_installation'] ?? 0;
		$rewriteAlias = $settings['rewrite_language_alias'] ?? 0;
		$rewriteUrl = $settings['rewrite_language_url'] ?? 0;
		$originalLang = $settings['language'] ?? 'en';
		
		if (! $rewriteAlias || ! $rewriteUrl) return;
		
		$requestUri = urldecode($_SERVER['REQUEST_URI']) ?? '';
		$requestUri = sanitize_text_field(wp_unslash($requestUri));
		
		// Remove subfolder if present
		if ($subfolder) {
			$homePath = wp_parse_url(home_url(), PHP_URL_PATH) ?? '';
			if ($homePath && strpos($requestUri, $homePath) === 0) {
				$requestUri = substr($requestUri, strlen($homePath));
			}
		}
		
		// Normalize path
		$rawUri = rtrim(home_url() . $requestUri, '/');
		$rawUriWithSlash = $rawUri . '/';
		
		$uri  = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
		$pattern = '#^/(' . implode ( '|', array_map ( 'preg_quote', $gpt_supported_languages ) ) . ')(/.*)?$#i';
		
		// Optional: remove subfolder if set
		$subfolder_prefix = ''; // To hold subfolder if any
		if ($settings['subfolder_installation']) {
			$uri_parts = explode('/', ltrim($uri, '/'));
			
			// Remove the first part (subfolder name)
			$subfolder_prefix = '/' . array_shift($uri_parts);
			
			// Rebuild URI without subfolder
			$uri_to_match = '/' . implode('/', $uri_parts);
		} else {
			$uri_to_match = $uri;
		}
		
		if (preg_match ( $pattern, $uri_to_match, $matches )) {
			$currentLang = strtolower ( $matches [1] );
		} else {
			return;
		}
			
		// Check translated alias
		$table = $wpdb->prefix . 'gptranslate';
		$query = $wpdb->prepare(
						"SELECT pagelink FROM $table WHERE published = 1 AND languagetranslated = %s AND (
						translated_alias = %s OR translated_alias = %s)",
						$currentLang, $rawUri, $rawUriWithSlash );
		
		$originalPageLink = $wpdb->get_var($query);
		
		if (! $originalPageLink) return;
		
		// Extract path from URL
		$parsed = wp_parse_url($originalPageLink);
		$originalPath = $parsed['path'] ?? '';
		
		// Remove subfolder prefix again from final path
		if ($subfolder && !empty($homePath) && strpos($originalPath, $homePath) === 0) {
			$originalPath = substr($originalPath, strlen($homePath));
			define('GPTRANSLATE_SUBFOLDER_REPLACED', true);
		}
		
		// Clean path
		$originalPath = preg_replace('#^/?index\.php/?#i', '', $originalPath);
		$originalPath = '/' . ltrim($originalPath, '/');
		
		// Preserve the public URL (as requested by the user) before rewriting to original WP path
		if (empty($GLOBALS['gptranslate_public_request_uri'])) {
			$GLOBALS['gptranslate_public_request_uri'] = isset($_SERVER['REQUEST_URI']) ? wp_unslash($_SERVER['REQUEST_URI']) : '';
		}
		
		// Rewrite the request
		$_SERVER['REQUEST_URI'] = $originalPath;
	}, 1);
	
	/**
	 * Detect language from URI and rewrite $_SERVER['REQUEST_URI'] accordingly defining a GPTRANSLATE_CURRENT_LANG constant
	 */
	add_action ( 'init', function () use ($gpt_supported_languages, $polylangActive, $settings) {
		if (is_admin()) return;
		
		$uri  = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
		$pattern = '#^/(' . implode ( '|', array_map ( 'preg_quote', $gpt_supported_languages ) ) . ')(/.*)?$#i';

		// Optional: remove subfolder if set
		$subfolder_prefix = ''; // To hold subfolder if any, avoid double removal
		if ($settings['subfolder_installation'] && !defined('GPTRANSLATE_SUBFOLDER_REPLACED')) {
			$uri_parts = explode('/', ltrim($uri, '/'));
			
			// Remove the first part (subfolder name)
			$subfolder_prefix = '/' . array_shift($uri_parts);
			
			// Rebuild URI without subfolder
			$uri_to_match = '/' . implode('/', $uri_parts);
		} else {
			$uri_to_match = $uri;
		}

		if (preg_match ( $pattern, $uri_to_match, $matches )) {
			$lang = strtolower ( $matches [1] );
			$new_uri = isset ( $matches [2] ) ? $matches [2] : '/';
			if ($settings ['rewrite_language_url'] == 1 && !$polylangActive) {
				$_SERVER ['REQUEST_URI'] = $subfolder_prefix . $new_uri;
			}

			if (! defined ( 'GPTRANSLATE_CURRENT_LANG' )) {
				define ( 'GPTRANSLATE_CURRENT_LANG', $lang );
			}
		}
	}, 2);
	
	/**
	 * Replace and manage the lang attribute of the html tag
	 */
	add_filter ( 'language_attributes', function ($output) use ($settings, $polylangActive) {
		if (! defined ( 'GPTRANSLATE_CURRENT_LANG' ) || (!$settings['set_html_lang'] && !$settings ['rewrite_language_url']) || $polylangActive)
			return $output;
		
		$altflags_array = get_option ( 'gptranslate_options' ) ['alt_flags'] ?? [ ];

		$lang_map = [ 
				// Variazioni per EN
				'en' => (in_array ( 'usa', $altflags_array ) || in_array ( 'canada', $altflags_array )) ? 'en-US' : 'en-GB',

				// Variazioni per FR
				'fr' => in_array ( 'quebec', $altflags_array ) ? 'fr-CA' : 'fr-FR',

				// Variazioni per PT
				'pt' => in_array ( 'brazil', $altflags_array ) ? 'pt-BR' : 'pt-PT',

				// Variazioni per ES
				'es' => in_array ( 'mexico', $altflags_array ) ? 'es-MX' : (in_array ( 'argentina', $altflags_array ) ? 'es-AR' : (in_array ( 'colombia', $altflags_array ) ? 'es-CO' : 'es-ES')),

				// Tutte le altre senza variazioni
				'af' => 'af-ZA',
				'sq' => 'sq-AL',
				'am' => 'am-ET',
				'ar' => 'ar-SA',
				'hy' => 'hy-AM',
				'az' => 'az-AZ',
				'eu' => 'eu-ES',
				'be' => 'be-BY',
				'bn' => 'bn-BD',
				'bs' => 'bs-BA',
				'bg' => 'bg-BG',
				'ca' => 'ca-ES',
				'ceb' => 'ceb-PH',
				'ny' => 'ny-MW',
				'zh' => 'zh-CN',
				'zt' => 'zh-TW',
				'co' => 'co-FR',
				'hr' => 'hr-HR',
				'cs' => 'cs-CZ',
				'da' => 'da-DK',
				'nl' => 'nl-NL',
				'eo' => 'eo-EO',
				'et' => 'et-EE',
				'tl' => 'tl-PH',
				'fi' => 'fi-FI',
				'fy' => 'fy-NL',
				'gl' => 'gl-ES',
				'ka' => 'ka-GE',
				'de' => 'de-DE',
				'el' => 'el-GR',
				'gu' => 'gu-IN',
				'ht' => 'ht-HT',
				'ha' => 'ha-NG',
				'haw' => 'haw-US',
				'iw' => 'iw-IL',
				'hi' => 'hi-IN',
				'hmn' => 'hmn-US',
				'hu' => 'hu-HU',
				'is' => 'is-IS',
				'ig' => 'ig-NG',
				'id' => 'id-ID',
				'ga' => 'ga-IE',
				'it' => 'it-IT',
				'ja' => 'ja-JP',
				'jw' => 'jw-ID',
				'kn' => 'kn-IN',
				'kk' => 'kk-KZ',
				'km' => 'km-KH',
				'ko' => 'ko-KR',
				'ku' => 'ku-TR',
				'ky' => 'ky-KG',
				'lo' => 'lo-LA',
				'la' => 'la-VA',
				'lv' => 'lv-LV',
				'lt' => 'lt-LT',
				'lb' => 'lb-LU',
				'mk' => 'mk-MK',
				'mg' => 'mg-MG',
				'ms' => 'ms-MY',
				'ml' => 'ml-IN',
				'mt' => 'mt-MT',
				'mi' => 'mi-NZ',
				'mr' => 'mr-IN',
				'mn' => 'mn-MN',
				'my' => 'my-MM',
				'ne' => 'ne-NP',
				'no' => 'no-NO',
				'ps' => 'ps-AF',
				'fa' => 'fa-IR',
				'pl' => 'pl-PL',
				'pa' => 'pa-IN',
				'ro' => 'ro-RO',
				'ru' => 'ru-RU',
				'sm' => 'sm-WS',
				'gd' => 'gd-GB',
				'sr' => 'sr-RS',
				'st' => 'st-LS',
				'sn' => 'sn-ZW',
				'sd' => 'sd-PK',
				'si' => 'si-LK',
				'sk' => 'sk-SK',
				'sl' => 'sl-SI',
				'so' => 'so-SO',
				'su' => 'su-ID',
				'sw' => 'sw-TZ',
				'sv' => 'sv-SE',
				'tg' => 'tg-TJ',
				'ta' => 'ta-IN',
				'te' => 'te-IN',
				'th' => 'th-TH',
				'tr' => 'tr-TR',
				'uk' => 'uk-UA',
				'ur' => 'ur-PK',
				'uz' => 'uz-UZ',
				'vi' => 'vi-VN',
				'cy' => 'cy-GB',
				'xh' => 'xh-ZA',
				'yi' => 'yi-US',
				'yo' => 'yo-NG',
				'zu' => 'zu-ZA'
		];

		$gpt_lang = strtolower ( GPTRANSLATE_CURRENT_LANG );
		$full_lang = $lang_map [$gpt_lang] ?? $gpt_lang;

		$output = 'lang="' . esc_attr ( $full_lang ) . '"';

		// --- RTL support: set WP_Locale direction dynamically ---
		global $wp_locale;
		$rtl_languages = array("ar", "fa", "iw", "ps", "sd", "ur", "yi"); // Arabic, Hebrew, Persian, Urdu
		if ( (int)$settings['auto_set_language_direction'] && in_array( $gpt_lang, $rtl_languages, true ) ) {
			$wp_locale->text_direction = 'rtl';
			// ensure HTML dir="rtl"
			if ( strpos( $output, 'dir=' ) === false ) {
				$output .= ' dir="rtl"';
			} else {
				$output = preg_replace( '/dir="ltr"/', 'dir="rtl"', $output );
			}
			return $output;
		}
		
		return $output;
	} );

	/**
	 * Inject alternate hreflang links in <head>
	 */
	add_action ( 'wp_head', function () use ($enabledLanguages, $polylangActive, $settings) {
		if (! defined ( 'GPTRANSLATE_CURRENT_LANG' ) || ! $settings ['add_alternate'] || ! $settings ['serverside_translations'] || $polylangActive) {
			return;
		}

		$current_lang = GPTRANSLATE_CURRENT_LANG;
		$request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
		$request_path = sanitize_text_field ( wp_unslash ( $request_uri ) );

		// Remove query string
		$request_path = strtok ( $request_path, '?' );
		
		// Optional subfolder cleanup
		if (!empty($settings['subfolder_installation'])) {
			$home_path = wp_parse_url(home_url(), PHP_URL_PATH) ?? '';
			if ($home_path && strpos($request_path, $home_path) === 0) {
				$request_path = substr($request_path, strlen($home_path));
			}
		}

		// Remove current language code from URL if present
		$current_path = preg_replace ( '#^/' . preg_quote ( $current_lang, '#' ) . '(/|$)#', '/', $request_path );

		$site_url = home_url ();

		foreach ( $enabledLanguages as $lang ) {
			$translated_url = rtrim ( $site_url, '/' ) . '/' . $lang . rtrim ( $current_path, '/' );

			echo '<link rel="alternate" hreflang="' . esc_attr ( $lang ) . '" href="' . esc_url ( $translated_url . '/' ) . "\" />\n";

			if ($lang === $settings ['language']) {
				echo '<link rel="alternate" hreflang="x-default" href="' . esc_url ( $translated_url . '/' ) . "\" />\n";
			}
		}
	} );
	
	/**
	 * Handle the multilanguage URLs
	 */
	if ($settings ['rewrite_language_url'] == 1 && !$polylangActive) {
		// Prevent WP to redirect
		add_filter ( 'redirect_canonical', function ($redirect_url, $requested_url) use ($gpt_supported_languages) {
			if (defined ( 'GPTRANSLATE_CURRENT_LANG' )) {
				return false;
			}
		}, 10, 2 );

		/**
		 * Add rewrite rules to allow /lang/ URLs
		 */
		add_filter ( 'rewrite_rules_array', function ($rules) use ($gpt_supported_languages) {
			$new_rules = [ ];
			foreach ( $gpt_supported_languages as $lang ) {
				$new_rules ["{$lang}/(.*)$"] = 'index.php?pagename=$matches[1]&gptlang=' . $lang;
				$new_rules ["{$lang}$"] = 'index.php?gptlang=' . $lang;
			}
			return $new_rules + $rules;
		} );

		/**
		 * Redirect URLs without language prefix to default language
		 */
		add_action ( 'template_redirect', function () use ($gpt_supported_languages, $settings) {
			$defaultOriginalLanguage = get_option ( 'gptranslate_options' ) ['language'] ?? 'en';

			$default_lang = $defaultOriginalLanguage;
			$uri  = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';

			// Exceptions and management for subfolder installations
			if($settings['subfolder_installation']) {
				// Explode URI into parts
				$uri_parts = explode('/', ltrim($uri, '/'));
				
				// Remove the first part (subfolder)
				array_shift($uri_parts);
				
				// Rebuild the URI
				$uri = '/' . implode('/', $uri_parts);
			}

			// Avoid multiple redirects in the same session
			if (! session_id ()) {
				session_start ();
			}

			if (! empty ( $_SESSION ['gptranslate_redirected'] ) && ($_SESSION ['gptranslate_redirected'] === $uri || $settings ['subfolder_installation'])) {
				unset ( $_SESSION ['gptranslate_redirected'] );
				return;
			}

			// Skip static assets and admin/API routes
			if (preg_match ( '#\.(ico|png|jpe?g|gif|svg|css|js|woff2?|ttf|eot|mp4|webm)$#i', $uri ) || strpos ( $uri, '/wp-' ) === 0 || strpos ( $uri, '/wp-json/' ) === 0) {
				return;
			}

			// If a language has already been detected and defined, skip redirect
			if (defined ( 'GPTRANSLATE_CURRENT_LANG' )) {
				return;
			}

			// Skip the default language if enabled
			if(isset($settings ['omit_prefix_original_language']) && $settings ['omit_prefix_original_language'] == 1) {
				return;
			}

			// Redirect to default language
			$_SESSION ['gptranslate_redirected'] = $uri;
			wp_redirect ( home_url ( "/{$default_lang}{$uri}" ), 301 );
			exit ();
		} );

		/**
		 * Add gptlang as a recognized query var
		 */
		add_filter ( 'query_vars', function ($vars) {
			$vars [] = 'gptlang';
			return $vars;
		} );

		/**
		 * Helper function to get current language
		 */
		function gpt_get_current_lang() {
			if (defined ( 'GPTRANSLATE_CURRENT_LANG' ))
				return GPTRANSLATE_CURRENT_LANG;
			$lang = get_query_var ( 'gptlang' );
			return $lang ?: 'en';
		}
		
		/**
		 * Rewrite internal links to include language prefix
		 */
		if ($settings ['rewrite_page_links'] == 1 && !$polylangActive) {
			add_action('template_redirect', function () use ($gpt_supported_languages, $settings) {
				ob_start(function ($html) use ($gpt_supported_languages, $settings) {
					$current_lang = defined('GPTRANSLATE_CURRENT_LANG') ? GPTRANSLATE_CURRENT_LANG : ($settings['language'] ?? 'en');
					
					$site_url = home_url();
					$parsed_site_url = wp_parse_url($site_url);
					$base_path = rtrim($parsed_site_url['path'] ?? '', '');
					
					return preg_replace_callback('#<a\s[^>]*href=["\']([^"\']+)["\']#i', function ($matches) use ($current_lang, $gpt_supported_languages, $parsed_site_url, $base_path, $site_url, $settings) {
						$original_url = $matches[1];
						$url = $original_url;
						
						// Skip image links (lightbox etc.)
						$image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'tiff'];
						$lower_url = strtolower(parse_url($url, PHP_URL_PATH));
						foreach ($image_extensions as $ext) {
							if (str_ends_with($lower_url, '.' . $ext)) {
								return $matches[0];
							}
						}
						
						// Skip external URLs
						if (preg_match('#^https?://#', $url) && strpos($url, $parsed_site_url['host']) === false) {
							return $matches[0];
						}
						
						// Skip admin-related URLs
						if (strpos($url, 'wp-admin') !== false || strpos($url, 'wp-login.php') !== false || strpos($url, 'wp-json') !== false) {
							return $matches[0];
						}
						
						// Skip adding prefix for default language if enabled
						$defaultOriginalLanguage = get_option ( 'gptranslate_options' ) ['language'] ?? 'en';
						if (isset($settings['omit_prefix_original_language']) && $settings['omit_prefix_original_language'] == 1 && $current_lang == $defaultOriginalLanguage) {
							return $matches[0];
						}

						// Convert absolute local URL to relative
						if (strpos($url, $parsed_site_url['scheme'] . '://' . $parsed_site_url['host']) === 0) {
							$url = substr($url, strlen($parsed_site_url['scheme'] . '://' . $parsed_site_url['host']));
						}
						
						// Remove base path if present
						if (!empty($base_path) && strpos($url, $base_path) === 0) {
							$url = substr($url, strlen($base_path));
						}
						
						// Ensure it starts with /
						if (strpos($url, '/') !== 0) {
							return $matches[0]; // ignore non-relative, non-local URLs
						}
						
						// Verifica se lo slug della lingua  presente, e se s, sostituiscilo con la lingua corrente
						foreach ($gpt_supported_languages as $lang) {
							if (preg_match("#^/{$lang}(/|\$)#i", $url)) {
								// Se la lingua trovata  diversa dalla lingua corrente, sostituire lo slug della lingua
								if ($lang !== $current_lang) {
									// Sostituisci lo slug con la lingua corrente
									$url = preg_replace("#^/{$lang}#", "/{$current_lang}", $url);
								}
								return str_replace($original_url, $url, $matches[0]);
							}
						}
						
						// Build new full absolute URL
						$new_url = rtrim($site_url, '/') . '/' . $current_lang . $url;
						
						return str_replace($original_url, $new_url, $matches[0]);
						
					}, $html);
				});
			}, 15);
		}
	}
	
	/**
	 * Add canonical link tag to the head
	 */
	add_action('wp_head', function() use ($settings) {
		
		// Check if canonical is enabled
		if (empty($settings['add_canonical'])) {
			return;
		}
		
		// Only apply to HTML pages (not feeds, REST API, etc.)
		if (is_feed() || is_robots() || is_trackback()) {
			return;
		}
		
		// ------------------------------------------------------------------
		// Get CURRENT PUBLIC URL (translated), not the rewritten WP one
		// ------------------------------------------------------------------
		$pageLink = '';
		
		$publicRequestUri = $GLOBALS['gptranslate_public_request_uri'] ?? '';
		
		if (!empty($publicRequestUri)) {
			
			// Build absolute URL from home_url() origin + original REQUEST_URI
			$homeParts = wp_parse_url(home_url());
			
			$origin  = ($homeParts['scheme'] ?? 'http') . '://' . ($homeParts['host'] ?? '');
			if (!empty($homeParts['port'])) {
				$origin .= ':' . $homeParts['port'];
			}
			
			$pageLink = $origin . $publicRequestUri;
			
			// Canonical should not include query string or fragment
			$parsed = wp_parse_url($pageLink);
			if (!empty($parsed['scheme']) && !empty($parsed['host'])) {
				$rebuilt  = $parsed['scheme'] . '://' . $parsed['host'];
				if (!empty($parsed['port'])) {
					$rebuilt .= ':' . $parsed['port'];
				}
				$rebuilt .= $parsed['path'] ?? '/';
				$pageLink = $rebuilt;
			}
			
			// WordPress-style trailing slash consistency
			if (function_exists('gpt_trailingslashit_url')) {
				$pageLink = gpt_trailingslashit_url($pageLink);
			}
			
		} else {
			// Fallback (should not happen, but safe)
			$pageLink = home_url(add_query_arg(null, null));
		}
		
		// Optional urldecode (user setting)
		if (!empty($settings['serverside_translations_urldecode'])) {
			$pageLink = urldecode($pageLink);
		}
		
		// ------------------------------------------------------------------
		// Remove existing canonical tags (Yoast, Rank Math, Core, etc.)
		// ------------------------------------------------------------------
		remove_action('wp_head', 'rel_canonical');
		remove_action('wp_head', 'wp_shortlink_wp_head');
		
		// ------------------------------------------------------------------
		// Remove canonical from Yoast SEO
		// ------------------------------------------------------------------
		if (defined('WPSEO_VERSION')) {
			remove_action('wpseo_head', ['WPSEO_Frontend', 'canonical'], 20);
			// Alternative methods for different Yoast versions
			if (class_exists('WPSEO_Frontend')) {
				$instance = WPSEO_Frontend::get_instance();
				remove_action('wp_head', [$instance, 'canonical'], 20);
			}
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from Rank Math SEO
		// ------------------------------------------------------------------
		if (class_exists('RankMath')) {
			remove_action('rank_math/head', 'rank_math_canonical');
			remove_action('rank_math/head', ['RankMath\Frontend\Head', 'canonical'], 20);
			if (function_exists('rank_math')) {
				remove_action('wp_head', [rank_math()->frontend, 'canonical'], 1);
			}
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from SureRank
		// ------------------------------------------------------------------
		if (class_exists('SureRank\Inc\Frontend\Canonical')) {
			remove_action('surerank_print_meta', [SureRank\Inc\Frontend\Canonical::get_instance(), 'print_canonical_url'], 1);
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from All in One SEO (AIOSEO)
		// ------------------------------------------------------------------
		if (function_exists('aioseo')) {
			remove_action('wp_head', 'aioseo_canonical_tag', 1);
			if (class_exists('AIOSEO\Plugin\Common\Models\Post')) {
				add_filter('aioseo_canonical_url', '__return_false');
			}
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from SEOPress
		// ------------------------------------------------------------------
		if (function_exists('seopress_init')) {
			remove_action('wp_head', 'seopress_canonical', 1);
			add_filter('seopress_titles_canonical', '__return_false');
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from The SEO Framework
		// ------------------------------------------------------------------
		if (function_exists('the_seo_framework')) {
			remove_action('wp_head', [the_seo_framework(), 'canonical_url'], 1);
			add_filter('the_seo_framework_rel_canonical_output', '__return_false');
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from Slim SEO
		// ------------------------------------------------------------------
		if (class_exists('SlimSEO\MetaTags\Canonical')) {
			add_filter('slim_seo_canonical_url', '__return_false');
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from Jetpack
		// ------------------------------------------------------------------
		if (class_exists('Jetpack')) {
			add_filter('jetpack_enable_canonical', '__return_false');
		}
		
		// ------------------------------------------------------------------
		// Remove canonical from WP Meta SEO
		// ------------------------------------------------------------------
		if (class_exists('WpMetaSeo')) {
			add_filter('wpms_canonical_url', '__return_false');
		}
		
		// ------------------------------------------------------------------
		// Generic filter to override any other canonical
		// ------------------------------------------------------------------
		add_filter('get_canonical_url', '__return_false', 999);
		add_filter('wpseo_canonical', '__return_false', 999);
		add_filter('aioseop_canonical_url', '__return_false', 999);
		
		// Output GPTranslate canonical
		echo '<link rel="canonical" href="' . esc_url($pageLink) . '" />' . "\n";
		
	}, 1);
		
	/**
	 * Remove canonical tags from other SEO plugins
	 * This runs early to ensure GPTranslate has priority
	 */
	add_action('template_redirect', function() use ($settings) {

		if (!empty($settings['add_canonical'])) {

			// Core
			remove_action('wp_head', 'rel_canonical');

			// Yoast SEO
			add_filter('wpseo_canonical', '__return_false');

			// Rank Math
			add_filter('rank_math/frontend/canonical', '__return_false');

			// All in One SEO
			add_filter('aioseop_canonical_url', '__return_false');
		}

	}, 1);
} // end is_admin check
