<?php
namespace FEACF\Utils;

class Helper {
    /**
     * Centralized check for frontend ACF eligibility.
     * Returns true if user is logged in, is on a single post/page,
     * and there is at least one non-empty ACF field group assigned.
     */
    public static function can_edit_frontend_acf() {
        if (!is_user_logged_in()) return false;
        if (!is_singular()) return false;
        if (!self::user_can_edit_post()) return false;
        if (!function_exists('acf_get_field_groups')) return false;

        $field_groups = acf_get_field_groups(['post_id' => get_the_ID()]);
        if (empty($field_groups)) return false;

        foreach ($field_groups as $group) {
            $fields = acf_get_fields($group['ID']);
            if (!empty($fields)) return true;
        }
        return false;
    }
     

    /**
     * Format value for ACF date_time_picker field to yyyy-MM-ddTHH:mm for input type="datetime-local"
     */
    public static function format_date_time_picker($value) {
        if (empty($value)) return '';
        $dt = strtotime($value);
        if ($dt) {
            return date('Y-m-d\TH:i', $dt);
        }
        return $value;
    }

    /**
     * Format value for ACF date_picker field to yyyy-MM-dd for input type="date"
     */
    public static function format_date_picker($value) {
        if (empty($value)) {
            return '';
        }

        // Clean up known trailing format hints like " m/d/y"
        $clean = preg_replace('/\s+[a-zA-Z\/]+$/', '', trim($value));

        // Define formats to test (add more if needed)
        $formats = [
            'Y-m-d',   // 2025-09-05
            'd/m/Y',   // 05/09/2025
            'd-m-Y',   // 05-09-2025
            'm/d/Y',   // 09/05/2025
            'm-d-Y',   // 09-05-2025
            'Y/m/d',   // 2025/09/05
            'Y-d-m',   // 2025-05-09 (edge case)
        ];

        // Try formats in order
        foreach ($formats as $fmt) {
            $dt = \DateTime::createFromFormat($fmt, $clean);

            // Check parsing success
            if ($dt instanceof \DateTime && $dt->format($fmt) === $clean) {
                return $dt->format('Y-m-d');
            }
        }

        // Fallback: try native parsing
        try {
            $dt = new \DateTime($clean);
            return $dt->format('Y-m-d');
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * Get supported field types for frontend editing. In future, this can be extended to pull from admin settings or user meta.
     *
     * @return array
     */
    public static function get_supported_fields() {
        $supported_fields = [
            'text', //done
            'textarea', //done
            'number', //done
            'range', //done
            'email', //done
            'url', //done
            'password', //done
            'image', //done
            'file', //Done
            'wysiwyg', //Done
            'oembed', //done
            'select', //done
            'checkbox', //done
            'radio', //done
            'button_group',//done
            'true_false',//done
            'page_link', //done
            'date_picker', //done
            'date_time_picker', 
            'time_picker', //done
            'color_picker',            
            // Add more field types as needed
        ];
        $supported_fields = apply_filters('feacf_supported_fields', $supported_fields);
        // This can be made dynamic in the future (e.g. from options, user meta, etc)
        return $supported_fields;
    }

    /**
     * Check if the current user can edit posts
     *
     * @return bool
     */
    public static function user_can_edit_post($post_id = null) {
        // return __return_true(); 
        // For testing, always return true
        // Uncomment below for production
        //return is_user_logged_in() && current_user_can('edit_posts');
        $pid = $post_id ? (int) $post_id : (int) get_the_ID();
        $can = is_user_logged_in() && current_user_can('edit_post', $pid); 
        /**
         * Filter whether the current user can edit the given post via FEACF.
         *
         * @param bool $can
         * @param int $pid
         */
        return (bool) apply_filters('feacf_user_can_edit_post', $can, $pid);
    }


    /**
     * Get current post ACF fields
     *
     * @return array
     */
    public static function get_current_post_acf_fields() {
        if (!function_exists('get_field_objects')) return [];
        $fields = get_field_objects(get_the_ID());
        return $fields ?: [];
    }

    /**
     * Sanitize ACF input data
     *
     * @param mixed $input
     * @return mixed
     */
    public static function sanitize_acf_input($input) {
        if (is_array($input)) {
            $out = [];
            foreach ($input as $k => $v) {
                $out[$k] = self::sanitize_acf_input($v);
            }
            return $out;
        }

        if (is_string($input)) {
            return \sanitize_text_field($input);
        }

        // Keep other scalar types and null as-is.
        if (is_scalar($input) || $input === null) {
            return $input;
        }

        // For any other types (e.g., objects), return an empty string.
        return '';
    }
    /**
     * Format oEmbed field value
     *
     * @param string $name
     * @param int $post_id
     * @return mixed
     */
    public static function format_oembed($name, $post_id) {
        return get_field($name, $post_id, false);
    }
}
