<?php
/**
 * Plugin Name: Edu Para Turismo - Token Gate
 * Description: Token único por comprador para liberar /candidatura com expiração e uso único.
 * Author: Edu Para Turismo
 */

if (!defined('ABSPATH')) exit;

class EduTokenGate {
    const TABLE = 'edu_tokens';
    const PAGE_SLUG = 'candidatura'; // slug exato da página
    const DEFAULT_EXPIRE_DAYS = 7;

    public static function init() {
        add_action('init', [__CLASS__, 'maybe_create_table']);
        add_action('template_redirect', [__CLASS__, 'gate_candidatura_page'], 1);

        // WooCommerce: gera token ao completar pedido (ou processing)
        add_action('woocommerce_order_status_completed', [__CLASS__, 'wc_generate_token_on_order'], 10, 1);
        add_action('woocommerce_order_status_processing', [__CLASS__, 'wc_generate_token_on_order'], 10, 1);

        // (Opcional) Shortcode para exibir o link de candidatura (para página obrigado)
        add_shortcode('edu_candidatura_link', [__CLASS__, 'shortcode_candidatura_link']);

        // Shortcode universal: botão/endpoint para "consumir" token (uso único) após finalizar candidatura
        add_shortcode('edu_consumir_token', [__CLASS__, 'shortcode_consumir_token']);

        // Endpoint para consumir token via POST
        add_action('admin_post_nopriv_edu_consumir_token', [__CLASS__, 'handle_consumir_token']);
        add_action('admin_post_edu_consumir_token', [__CLASS__, 'handle_consumir_token']);

        // Integração tentativa com WPUF (se existir): consumir token após envio
        add_action('wpuf_add_post_after_insert', [__CLASS__, 'wpuf_after_submit_consume_token'], 10, 4);
        add_action('wpuf_after_create_post', [__CLASS__, 'wpuf_after_submit_consume_token_alt'], 10, 3);
    }

    public static function db_table_name() {
        global $wpdb;
        return $wpdb->prefix . self::TABLE;
    }

    public static function maybe_create_table() {
        global $wpdb;
        $table = self::db_table_name();

        // Evita rodar toda hora
        $opt = get_option('edu_tokens_table_ready');
        if ($opt === '1') return;

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            token VARCHAR(80) NOT NULL,
            email VARCHAR(190) NOT NULL,
            order_id BIGINT UNSIGNED NULL,
            created_at DATETIME NOT NULL,
            expires_at DATETIME NOT NULL,
            used_at DATETIME NULL,
            used_ip VARCHAR(45) NULL,
            used_ua VARCHAR(255) NULL,
            notes TEXT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY token (token),
            KEY email (email),
            KEY order_id (order_id),
            KEY expires_at (expires_at),
            KEY used_at (used_at)
        ) {$charset};";

        dbDelta($sql);

        update_option('edu_tokens_table_ready', '1');
    }

    // ---------------------------
    // GATE na página /candidatura
    // ---------------------------
    public static function gate_candidatura_page() {
        if (!is_page(self::PAGE_SLUG)) return;

        $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
        if (empty($token)) {
            self::deny("Acesso restrito. Use o link enviado após a compra.");
        }

        $row = self::get_token_row($token);
        if (!$row) {
            self::deny("Token inválido.");
        }

        $now = current_time('mysql');
        if ($row->used_at !== null && $row->used_at !== '0000-00-00 00:00:00') {
            self::deny("Este token já foi utilizado.");
        }

        if ($row->expires_at < $now) {
            self::deny("Token expirado.");
        }

        // Guarda token na sessão do WP (cookie) para facilitar uso no formulário/shortcode
        setcookie('edu_token', $token, time() + 3600 * 6, COOKIEPATH ?: '/', COOKIE_DOMAIN, is_ssl(), true);
        $_COOKIE['edu_token'] = $token;
    }

    private static function deny($msg) {
        // Você pode trocar home_url() por uma página "acesso negado"
        wp_redirect(home_url('/?acesso=candidatura_negado&motivo=' . rawurlencode($msg)));
        exit;
    }

    // ---------------------------
    // WooCommerce: gerar token
    // ---------------------------
    public static function wc_generate_token_on_order($order_id) {
        if (!function_exists('wc_get_order')) return;
        $order = wc_get_order($order_id);
        if (!$order) return;

        // Evita duplicar token no mesmo pedido
        $existing = $order->get_meta('_edu_token');
        if (!empty($existing)) return;

        $email = $order->get_billing_email();
        if (empty($email)) return;

        $token = self::generate_unique_token();
        $created_at = current_time('mysql');

        $expires_at = date('Y-m-d H:i:s', strtotime($created_at . ' +' . self::DEFAULT_EXPIRE_DAYS . ' days'));

        self::insert_token($token, $email, $order_id, $created_at, $expires_at);

        // salva no pedido
        $order->update_meta_data('_edu_token', $token);
        $order->save();

        // envia email simples com link
        $link = add_query_arg(['token' => $token], home_url('/' . self::PAGE_SLUG . '/'));
        self::send_token_email($email, $link, $token);
    }

    private static function send_token_email($email, $link, $token) {
        $subject = 'Acesso à Candidatura - Edu Para Turismo';
        $message = "Olá!\n\nSegue seu link exclusivo de acesso à página de candidatura:\n{$link}\n\n"
                 . "Este link expira em " . self::DEFAULT_EXPIRE_DAYS . " dias e pode ser usado apenas uma vez (após finalizar a candidatura).\n\n"
                 . "Se precisar de ajuda, responda este e-mail.\n\nEdu Para Turismo";
        wp_mail($email, $subject, $message);
    }

    // ---------------------------
    // Consumir token (uso único)
    // ---------------------------
    public static function shortcode_consumir_token($atts) {
        // Este shortcode cria um botão que consome o token (uso único).
        // Você coloca esse botão NO FINAL da página de candidatura (Elementor HTML widget)
        $token = self::get_current_token();
        if (!$token) return '<div style="padding:12px;border:1px solid #eee;border-radius:8px;">Token não detectado. Acesse pelo link exclusivo.</div>';

        $action = esc_url(admin_url('admin-post.php'));
        $nonce = wp_create_nonce('edu_consumir_token');

        return '
        <form method="post" action="'.$action.'" style="margin-top:16px;">
            <input type="hidden" name="action" value="edu_consumir_token">
            <input type="hidden" name="edu_token" value="'.esc_attr($token).'">
            <input type="hidden" name="edu_nonce" value="'.esc_attr($nonce).'">
            <button type="submit" style="padding:12px 16px;border-radius:10px;border:0;background:#1a7f37;color:#fff;font-weight:700;cursor:pointer;">
                Finalizar candidatura (confirmar envio)
            </button>
        </form>';
    }

    public static function handle_consumir_token() {
        $nonce = $_POST['edu_nonce'] ?? '';
        if (!wp_verify_nonce($nonce, 'edu_consumir_token')) {
            wp_die('Nonce inválido.');
        }

        $token = isset($_POST['edu_token']) ? sanitize_text_field($_POST['edu_token']) : '';
        if (empty($token)) wp_die('Token ausente.');

        // valida novamente
        $row = self::get_token_row($token);
        if (!$row) wp_die('Token inválido.');
        $now = current_time('mysql');

        if ($row->used_at) wp_die('Token já utilizado.');
        if ($row->expires_at < $now) wp_die('Token expirado.');

        self::mark_token_used($token);

        // Redireciona para obrigado
        wp_redirect(home_url('/obrigado-candidatura/'));
        exit;
    }

    // ---------------------------
    // WPUF: tentar consumir token após enviar
    // (se você usa WPUF para candidatura)
    // ---------------------------
    public static function wpuf_after_submit_consume_token($post_id, $form_id, $form_settings, $meta) {
        $token = self::get_current_token();
        if (!$token) return;
        self::safe_consume_if_valid($token);
    }
    public static function wpuf_after_submit_consume_token_alt($post_id, $form_id, $form_settings) {
        $token = self::get_current_token();
        if (!$token) return;
        self::safe_consume_if_valid($token);
    }

    private static function safe_consume_if_valid($token) {
        $row = self::get_token_row($token);
        if (!$row) return;

        $now = current_time('mysql');
        if ($row->used_at) return;
        if ($row->expires_at < $now) return;

        self::mark_token_used($token);
    }

    // ---------------------------
    // Shortcode link candidatura
    // ---------------------------
    public static function shortcode_candidatura_link() {
        // Se a pessoa estiver logada e tiver um token cookie, mostra link
        $token = self::get_current_token();
        if (!$token) return '';

        $link = add_query_arg(['token' => $token], home_url('/' . self::PAGE_SLUG . '/'));
        return '<a href="'.esc_url($link).'" style="font-weight:700;">Acessar página de candidatura</a>';
    }

    // ---------------------------
    // Helpers DB
    // ---------------------------
    private static function get_token_row($token) {
        global $wpdb;
        $table = self::db_table_name();
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE token = %s LIMIT 1", $token));
    }

    private static function insert_token($token, $email, $order_id, $created_at, $expires_at) {
        global $wpdb;
        $table = self::db_table_name();
        $wpdb->insert($table, [
            'token' => $token,
            'email' => $email,
            'order_id' => $order_id,
            'created_at' => $created_at,
            'expires_at' => $expires_at,
            'used_at' => null
        ], ['%s','%s','%d','%s','%s','%s']);
    }

    private static function mark_token_used($token) {
        global $wpdb;
        $table = self::db_table_name();

        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $now = current_time('mysql');

        $wpdb->update($table, [
            'used_at' => $now,
            'used_ip' => substr($ip, 0, 45),
            'used_ua' => substr($ua, 0, 255),
        ], [
            'token' => $token
        ], ['%s','%s','%s'], ['%s']);
    }

    private static function generate_unique_token() {
        // token curto e forte
        // exemplo: EDU-9f2c1a... (evita caracteres chatos)
        do {
            $raw = bin2hex(random_bytes(16)); // 32 chars
            $token = 'EDU-' . $raw;
        } while (self::token_exists($token));
        return $token;
    }

    private static function token_exists($token) {
        global $wpdb;
        $table = self::db_table_name();
        $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$table} WHERE token = %s LIMIT 1", $token));
        return !empty($exists);
    }

    private static function get_current_token() {
        // prioridade: URL, depois cookie
        if (!empty($_GET['token'])) return sanitize_text_field($_GET['token']);
        if (!empty($_COOKIE['edu_token'])) return sanitize_text_field($_COOKIE['edu_token']);
        return '';
    }
}

EduTokenGate::init();
