<?php
/**
 * Plugin Name: Edu Para Turismo - Candidaturas (CPT + ACF + Form)
 * Description: Cria CPT de candidaturas e formulário frontend via ACF (sem Elementor forms), com uploads e status pendente.
 * Version: 1.0.0
 */

if (!defined('ABSPATH')) exit;

/**
 * 1) Registra CPT: candidatura_agente
 */
add_action('init', function () {
  $labels = array(
    'name'               => 'Candidaturas',
    'singular_name'      => 'Candidatura',
    'menu_name'          => 'Candidaturas',
    'name_admin_bar'     => 'Candidatura',
    'add_new'            => 'Nova',
    'add_new_item'       => 'Nova candidatura',
    'new_item'           => 'Nova candidatura',
    'edit_item'          => 'Editar candidatura',
    'view_item'          => 'Ver candidatura',
    'all_items'          => 'Todas as candidaturas',
    'search_items'       => 'Buscar candidaturas',
    'not_found'          => 'Nenhuma candidatura encontrada',
    'not_found_in_trash' => 'Nenhuma candidatura na lixeira',
  );

  $args = array(
    'labels'             => $labels,
    'public'             => false,
    'show_ui'            => true,
    'show_in_menu'       => true,
    'menu_position'      => 25,
    'menu_icon'          => 'dashicons-id',
    'supports'           => array('title', 'author'),
    'capability_type'    => 'post',
    'map_meta_cap'       => true,
    'has_archive'        => false,
    'rewrite'            => false,
  );

  register_post_type('candidatura_agente', $args);
});

/**
 * 2) Field Group ACF (via código) - carrega após ACF iniciar
 */
add_action('acf/init', function () {
  if (!function_exists('acf_add_local_field_group')) return;

  acf_add_local_field_group(array(
    'key' => 'group_edu_candidatura',
    'title' => 'Candidatura — Agente Regional',
    'fields' => array(

      // ====== Identificação ======
      array(
        'key' => 'field_cand_nome',
        'label' => 'Nome completo',
        'name' => 'cand_nome',
        'type' => 'text',
        'required' => 1,
      ),
      array(
        'key' => 'field_cand_email',
        'label' => 'E-mail',
        'name' => 'cand_email',
        'type' => 'email',
        'required' => 1,
      ),
      array(
        'key' => 'field_cand_whats',
        'label' => 'WhatsApp / Telefone',
        'name' => 'cand_whats',
        'type' => 'text',
        'required' => 1,
        'instructions' => 'Ex.: (91) 99999-9999',
      ),

      // Taxonomia Cidade
      array(
        'key' => 'field_cand_cidade',
        'label' => 'Cidade de atuação',
        'name' => 'cand_cidade',
        'type' => 'taxonomy',
        'taxonomy' => 'cidade',
        'field_type' => 'select',
        'allow_null' => 0,
        'add_term' => 0,
        'save_terms' => 0,
        'load_terms' => 0,
        'return_format' => 'id',
        'required' => 1,
      ),

      // Taxonomia Polo
      array(
        'key' => 'field_cand_polo',
        'label' => 'Polo turístico',
        'name' => 'cand_polo',
        'type' => 'taxonomy',
        'taxonomy' => 'polo_turistico',
        'field_type' => 'select',
        'allow_null' => 0,
        'add_term' => 0,
        'save_terms' => 0,
        'load_terms' => 0,
        'return_format' => 'id',
        'required' => 1,
      ),

      // ====== Qualificação ======
      array(
        'key' => 'field_cand_bio',
        'label' => 'Experiência e qualificação',
        'name' => 'cand_bio',
        'type' => 'textarea',
        'required' => 1,
        'instructions' => 'Conte sua experiência com turismo, comunicação, vendas, conteúdo ou atuação local.',
        'rows' => 5,
      ),
      array(
        'key' => 'field_cand_portfolio',
        'label' => 'Portfólio (link) — opcional',
        'name' => 'cand_portfolio',
        'type' => 'url',
        'required' => 0,
      ),
      array(
        'key' => 'field_cand_equip_pc',
        'label' => 'Você possui notebook/PC para redação e trabalho?',
        'name' => 'cand_equip_pc',
        'type' => 'true_false',
        'ui' => 1,
        'required' => 1,
        'default_value' => 1,
      ),
      array(
        'key' => 'field_cand_equip_cel',
        'label' => 'Você possui celular com câmera em boas condições?',
        'name' => 'cand_equip_cel',
        'type' => 'true_false',
        'ui' => 1,
        'required' => 1,
        'default_value' => 1,
      ),

      // ====== Materiais obrigatórios ======
      array(
        'key' => 'field_cand_video_pessoal_upload',
        'label' => 'Vídeo de apresentação pessoal (upload) — opcional',
        'name' => 'cand_video_pessoal_upload',
        'type' => 'file',
        'return_format' => 'array',
        'library' => 'all',
        'mime_types' => 'mp4,mov,webm',
        'required' => 0,
        'instructions' => 'Se preferir, envie por link no campo abaixo (recomendado para arquivos grandes).',
      ),
      array(
        'key' => 'field_cand_video_pessoal_link',
        'label' => 'Vídeo de apresentação pessoal (link) — obrigatório se não enviar upload',
        'name' => 'cand_video_pessoal_link',
        'type' => 'url',
        'required' => 0,
        'instructions' => 'YouTube NÃO LISTADO / Google Drive / Dropbox (link público).',
      ),

      array(
        'key' => 'field_cand_video_local_upload',
        'label' => 'Vídeo de 1 minuto do local (upload) — opcional',
        'name' => 'cand_video_local_upload',
        'type' => 'file',
        'return_format' => 'array',
        'library' => 'all',
        'mime_types' => 'mp4,mov,webm',
        'required' => 0,
        'instructions' => 'Se preferir, envie por link no campo abaixo.',
      ),
      array(
        'key' => 'field_cand_video_local_link',
        'label' => 'Vídeo de 1 minuto do local (link) — obrigatório se não enviar upload',
        'name' => 'cand_video_local_link',
        'type' => 'url',
        'required' => 0,
        'instructions' => 'YouTube NÃO LISTADO / Google Drive / Dropbox (link público).',
      ),

      array(
        'key' => 'field_cand_texto_exemplo',
        'label' => 'Texto exemplo de matéria (obrigatório)',
        'name' => 'cand_texto_exemplo',
        'type' => 'textarea',
        'required' => 1,
        'instructions' => 'Escreva uma matéria curta (ex.: balneário, praia, igarapé, praça, evento).',
        'rows' => 10,
      ),

      array(
        'key' => 'field_cand_curriculo',
        'label' => 'Currículo (PDF)',
        'name' => 'cand_curriculo',
        'type' => 'file',
        'return_format' => 'array',
        'library' => 'all',
        'mime_types' => 'pdf',
        'required' => 1,
      ),

      array(
        'key' => 'field_cand_docs_extra',
        'label' => 'Documentos complementares (opcional)',
        'name' => 'cand_docs_extra',
        'type' => 'repeater',
        'required' => 0,
        'min' => 0,
        'layout' => 'table',
        'button_label' => 'Adicionar documento',
        'sub_fields' => array(
          array(
            'key' => 'field_cand_doc_file',
            'label' => 'Arquivo',
            'name' => 'arquivo',
            'type' => 'file',
            'return_format' => 'array',
            'library' => 'all',
            'mime_types' => 'pdf,jpg,jpeg,png',
            'required' => 1,
          )
        ),
      ),

      // ====== Compromissos ======
      array(
        'key' => 'field_cand_pj_compromisso',
        'label' => 'Compromisso de formalização PJ após aprovação',
        'name' => 'cand_pj_compromisso',
        'type' => 'true_false',
        'ui' => 1,
        'required' => 1,
        'instructions' => 'Declaro que tenho capacidade e disposição para me formalizar como PJ após aprovação.',
      ),
      array(
        'key' => 'field_cand_exclusividade',
        'label' => 'Compromisso de exclusividade territorial',
        'name' => 'cand_exclusividade',
        'type' => 'true_false',
        'ui' => 1,
        'required' => 1,
        'instructions' => 'Declaro ciência e concordância com exclusividade territorial definida em contrato.',
      ),
      array(
        'key' => 'field_cand_metas',
        'label' => 'Compromisso com metas mínimas',
        'name' => 'cand_metas',
        'type' => 'true_false',
        'ui' => 1,
        'required' => 1,
        'instructions' => 'Declaro ciência e concordância com metas mínimas de atuação (conteúdo e presença).',
      ),
      array(
        'key' => 'field_cand_lgpd',
        'label' => 'Consentimento de tratamento de dados (LGPD)',
        'name' => 'cand_lgpd',
        'type' => 'true_false',
        'ui' => 1,
        'required' => 1,
        'instructions' => 'Autorizo o uso dos dados enviados exclusivamente para fins de avaliação da candidatura.',
      ),

    ),
    'location' => array(
      array(
        array(
          'param' => 'post_type',
          'operator' => '==',
          'value' => 'candidatura_agente',
        ),
      ),
    ),
    'position' => 'normal',
    'style' => 'default',
    'label_placement' => 'top',
    'instruction_placement' => 'label',
    'hide_on_screen' => array(),
    'active' => true,
  ));
});

/**
 * 3) Shortcode do formulário: [edu_candidatura_form]
 * - Cria post pendente
 * - Usa ACF Form
 */
add_shortcode('edu_candidatura_form', function () {
  if (!function_exists('acf_form')) {
    return '<p><strong>Erro:</strong> ACF não carregou. Verifique o plugin ACF PRO.</p>';
  }

  // Mensagem acima do form
  $html = '';
  $html .= '<div class="edu-candidatura-wrap">';
  $html .= '<p><strong>Aviso:</strong> envie sua candidatura completa. Candidaturas incompletas são desconsideradas.</p>';

  // Config do form
  ob_start();

  acf_form(array(
    'post_id' => 'new_post',
    'new_post' => array(
      'post_type'   => 'candidatura_agente',
      'post_status' => 'pending',
    ),
    'post_title' => true, // vamos sobrescrever depois no hook, mas pode deixar
    'post_content' => false,
    'field_groups' => array('group_edu_candidatura'),
    'submit_value' => 'Enviar candidatura para análise',
    'updated_message' => 'Candidatura enviada com sucesso. Em breve você receberá retorno por e-mail.',
    'uploader' => 'wp',
    'honeypot' => true,
    'html_submit_button' => '<button class="acf-button button button-primary button-large" type="submit">%s</button>',
  ));

  $html .= ob_get_clean();
  $html .= '</div>';

  // CSS simples (opcional)
  $html .= '<style>
    .edu-candidatura-wrap{max-width:900px;margin:0 auto}
    .edu-candidatura-wrap .acf-form{margin-top:16px}
    .edu-candidatura-wrap .acf-fields>.acf-field{border:0;padding:10px 0}
    .edu-candidatura-wrap .acf-button{padding:14px 18px;font-size:16px}
  </style>';

  return $html;
});

/**
 * 4) Validações: exige vídeo por upload OU link (nos dois casos)
 */
add_action('acf/validate_save_post', function () {
  // Só validar quando estiver salvando candidatura via ACF form
  // Se estiver no admin editando, deixa passar
  if (is_admin()) return;

  // Checa se o POST tem o nosso grupo
  if (empty($_POST['acf'])) return;

  $acf = $_POST['acf'];

  // Helpers para pegar valor por key (ACF envia por field key)
  $get = function($field_key) use ($acf) {
    return isset($acf[$field_key]) ? $acf[$field_key] : null;
  };

  // Vídeo pessoal: upload OU link
  $pessoal_upload = $get('field_cand_video_pessoal_upload');
  $pessoal_link   = $get('field_cand_video_pessoal_link');

  if (empty($pessoal_upload) && empty($pessoal_link)) {
    acf_add_validation_error('field_cand_video_pessoal_link', 'Envie o vídeo pessoal por upload OU informe um link.');
  }

  // Vídeo local: upload OU link
  $local_upload = $get('field_cand_video_local_upload');
  $local_link   = $get('field_cand_video_local_link');

  if (empty($local_upload) && empty($local_link)) {
    acf_add_validation_error('field_cand_video_local_link', 'Envie o vídeo do local por upload OU informe um link.');
  }
});

/**
 * 5) Pós-salvamento: define título padrão + notifica admin
 */
add_action('acf/save_post', function ($post_id) {
  if (get_post_type($post_id) !== 'candidatura_agente') return;

  // Pega campos (pelo name)
  $nome   = get_field('cand_nome', $post_id);
  $email  = get_field('cand_email', $post_id);

  $cidade_id = get_field('cand_cidade', $post_id);
  $polo_id   = get_field('cand_polo', $post_id);

  $cidade = $cidade_id ? get_term($cidade_id) : null;
  $polo   = $polo_id ? get_term($polo_id) : null;

  $titulo = trim($nome);
  if ($cidade && !is_wp_error($cidade)) $titulo .= ' — ' . $cidade->name;
  if ($polo && !is_wp_error($polo)) $titulo .= ' (' . $polo->name . ')';

  // Atualiza título do post
  remove_action('acf/save_post', __FUNCTION__, 20);
  wp_update_post(array(
    'ID' => $post_id,
    'post_title' => $titulo,
  ));
  add_action('acf/save_post', __FUNCTION__, 20);

  // Notifica admin (simples)
  $admin_email = get_option('admin_email');
  $subject = '[Edu Para Turismo] Nova candidatura recebida';
  $message = "Uma nova candidatura foi enviada.\n\n";
  $message .= "Nome: {$nome}\n";
  $message .= "E-mail: {$email}\n";
  if ($cidade && !is_wp_error($cidade)) $message .= "Cidade: {$cidade->name}\n";
  if ($polo && !is_wp_error($polo)) $message .= "Polo: {$polo->name}\n";
  $message .= "\nAcesse o WP Admin para revisar (status: pendente).\n";

  @wp_mail($admin_email, $subject, $message);

}, 20);
