<?php
/**
 * Plugin Name: Anúncio Shortcodes (ACF + Taxonomias)
 * Description: Shortcodes para Elementor puxar terms/taxonomias e campos ACF do CPT anúncio.
 * Author: Portal
 * Version: 1.0.0
 */

if (!defined('ABSPATH')) exit;

/**
 * Helpers
 */
function portal_get_current_post_id_from_context($atts = []) {
  $post_id = 0;

  if (!empty($atts['post_id'])) {
    $post_id = (int) $atts['post_id'];
    if ($post_id > 0) return $post_id;
  }

  // Elementor/Loop/Single normalmente define global $post.
  $post_id = get_the_ID();
  if ($post_id) return (int)$post_id;

  // Fallback (raríssimo)
  global $post;
  if (!empty($post->ID)) return (int)$post->ID;

  return 0;
}

function portal_terms_to_html($terms, $style = 'list', $separator = ', ') {
  if (empty($terms) || is_wp_error($terms)) return '';

  $items = [];
  foreach ($terms as $t) {
    $name = esc_html($t->name);
    $url  = get_term_link($t);
    $url  = is_wp_error($url) ? '' : esc_url($url);

    $items[] = $url ? '<a href="'.$url.'">'.$name.'</a>' : $name;
  }

  if ($style === 'chips') {
    $out = '<div class="portal-terms portal-terms--chips">';
    foreach ($items as $html) {
      $out .= '<span class="portal-chip">'.$html.'</span>';
    }
    $out .= '</div>';
    return $out;
  }

  if ($style === 'ul') {
    $out = '<ul class="portal-terms portal-terms--ul">';
    foreach ($items as $html) {
      $out .= '<li>'.$html.'</li>';
    }
    $out .= '</ul>';
    return $out;
  }

  // default "list" inline
  return '<div class="portal-terms portal-terms--list">'.implode(esc_html($separator), $items).'</div>';
}

/**
 * Shortcode: [anuncio_terms tax="servicos" style="chips" sep=", " link="1" post_id=""]
 * - tax: slug da taxonomia (ex: servicos, cidades, polo_turistico, categoria_anuncio, destino_turistico)
 * - style: list | chips | ul
 * - sep: separador do style=list
 * - link: 1 (linka para arquivo da taxonomia) / 0 (sem link)
 */
add_shortcode('anuncio_terms', function($atts) {
  $atts = shortcode_atts([
    'tax'     => '',
    'style'   => 'chips',
    'sep'     => ', ',
    'link'    => '1',
    'post_id' => '',
    'prefix'  => '',
    'empty'   => '',
  ], (array)$atts);

  $tax = sanitize_key($atts['tax']);
  if (!$tax) return '';

  $post_id = portal_get_current_post_id_from_context($atts);
  if (!$post_id) return '';

  $terms = get_the_terms($post_id, $tax);
  if (empty($terms) || is_wp_error($terms)) {
    return $atts['empty'] !== '' ? esc_html($atts['empty']) : '';
  }

  // Se link=0, remove link
  if ((string)$atts['link'] === '0') {
    foreach ($terms as $t) {
      $t->link = '';
    }
    // portal_terms_to_html usa get_term_link; então vamos render sem link manual:
    $names = array_map(function($t){ return esc_html($t->name); }, $terms);
    if ($atts['style'] === 'chips') {
      $out = '<div class="portal-terms portal-terms--chips">';
      foreach ($names as $n) $out .= '<span class="portal-chip">'.$n.'</span>';
      $out .= '</div>';
      return ($atts['prefix'] ? '<div class="portal-prefix">'.esc_html($atts['prefix']).'</div>' : '').$out;
    }
    if ($atts['style'] === 'ul') {
      $out = '<ul class="portal-terms portal-terms--ul">';
      foreach ($names as $n) $out .= '<li>'.$n.'</li>';
      $out .= '</ul>';
      return ($atts['prefix'] ? '<div class="portal-prefix">'.esc_html($atts['prefix']).'</div>' : '').$out;
    }
    return ($atts['prefix'] ? '<div class="portal-prefix">'.esc_html($atts['prefix']).'</div>' : '').
      '<div class="portal-terms portal-terms--list">'.implode(esc_html($atts['sep']), $names).'</div>';
  }

  $html = portal_terms_to_html($terms, $atts['style'], $atts['sep']);
  if (!$html) return '';

  $prefix = $atts['prefix'] ? '<div class="portal-prefix">'.esc_html($atts['prefix']).'</div>' : '';
  return $prefix.$html;
});

/**
 * Shortcode: [anuncio_acf key="whatsapp" label="WhatsApp" format="1" empty=""]
 * - key: nome do campo ACF (field name)
 * - label: opcional para imprimir antes
 * - format: 1 (ACF formatado) / 0 (raw)
 * - empty: texto se vazio
 */
add_shortcode('anuncio_acf', function($atts) {
  $atts = shortcode_atts([
    'key'     => '',
    'label'   => '',
    'format'  => '1',
    'post_id' => '',
    'empty'   => '',
    'wrap'    => 'div', // div|span|p
  ], (array)$atts);

  $key = sanitize_key($atts['key']);
  if (!$key) return '';

  $post_id = portal_get_current_post_id_from_context($atts);
  if (!$post_id) return '';

  if (!function_exists('get_field')) {
    return ''; // ACF não carregou
  }

  $format = ((string)$atts['format'] === '0') ? false : true;
  $value  = get_field($key, $post_id, $format);

  if ($value === null || $value === '' || $value === false || (is_array($value) && empty($value))) {
    return $atts['empty'] !== '' ? esc_html($atts['empty']) : '';
  }

  // Render seguro
  if (is_array($value)) {
    // para campos tipo repeater/array, imprime como JSON legível (mínimo)
    $value = wp_json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  }

  $label = $atts['label'] ? '<strong class="portal-label">'.esc_html($atts['label']).':</strong> ' : '';
  $tag   = in_array($atts['wrap'], ['div','span','p'], true) ? $atts['wrap'] : 'div';

  return '<'.$tag.' class="portal-acf portal-acf--'.$key.'">'.$label.nl2br(esc_html((string)$value)).'</'.$tag.'>';
});

/**
 * Shortcode: [anuncio_whatsapp key="whatsapp" text="Falar no WhatsApp"]
 * Gera link wa.me formatando número.
 */
add_shortcode('anuncio_whatsapp', function($atts){
  $atts = shortcode_atts([
    'key'     => 'whatsapp',
    'text'    => 'Falar no WhatsApp',
    'post_id' => '',
    'class'   => 'portal-btn-whatsapp',
  ], (array)$atts);

  $post_id = portal_get_current_post_id_from_context($atts);
  if (!$post_id || !function_exists('get_field')) return '';

  $raw = get_field(sanitize_key($atts['key']), $post_id, false);
  if (!$raw) return '';

  // Mantém só números
  $num = preg_replace('/\D+/', '', (string)$raw);

  // Se veio sem DDI, assume BR (55)
  if (strlen($num) <= 11) $num = '55'.$num;

  $url = 'https://wa.me/'.rawurlencode($num);
  return '<a class="'.esc_attr($atts['class']).'" href="'.esc_url($url).'" target="_blank" rel="noopener">'.esc_html($atts['text']).'</a>';
});

/**
 * CSS básico para chips/botão.
 */
add_action('wp_enqueue_scripts', function(){
  $css = '
  .portal-terms--chips{display:flex;flex-wrap:wrap;gap:8px;align-items:center}
  .portal-chip{display:inline-flex;align-items:center;padding:6px 10px;border:1px solid rgba(0,0,0,.12);border-radius:999px;background:#fff;font-size:14px;line-height:1}
  .portal-chip a{text-decoration:none}
  .portal-btn-whatsapp{display:inline-block;padding:10px 14px;border-radius:10px;background:#25D366;color:#fff;text-decoration:none;font-weight:600}
  .portal-btn-whatsapp:hover{filter:brightness(.95)}
  .portal-prefix{margin-bottom:8px;font-weight:700}
  ';
  wp_register_style('portal-anuncio-shortcodes', false);
  wp_enqueue_style('portal-anuncio-shortcodes');
  wp_add_inline_style('portal-anuncio-shortcodes', $css);
});
